﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：カラー画像に最適化オプションを適用し、PDFを軽量化します。
        最適化をする画像のフィルター種類を選択します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class OptimizeImageSetValidFilter
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: OptimizeImageSetValidFilter in-pdf-file" +
                               " out-pdf-file");
            Console.WriteLine("ASCII85Decode ASCIIHexDecode CCITTFaxDecode DCTDecode");
            Console.WriteLine("FlateDecode JBIG2Decode JPXDecode LZWDecode RunLengthDecode");
            Console.WriteLine("各フィルター 0と1でそれぞれの有効を切替え：");
            Console.WriteLine("0:有効にしない 1:有効にする");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 11)
            {
                printUsage();
                return;
            }

            // コマンドライン引数の取得
            bool[] validFilter = new bool[9];
            try
            {
                for (int i = 0; i < 9; i++)
                {
                    validFilter[i] =
                        readBoolArgs(args[i + 2], "各フィルターの数値 は 0か1で指定してください。");
                }
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // 画像の最適化を行います。
                    optimizeDocSetValidFilter(doc, validFilter);

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void optimizeDocSetValidFilter(PtlPDFDocument doc,
                                                     bool[] validFilter)
        {
            using (PtlParamOptimize paramOptimize = new PtlParamOptimize())
            // 画像最適化パラメーターの取得
            // カラー画像最適化パラメーターの取得
            using (PtlParamOptimizeImage paramOptimizeImage =
            paramOptimize.getParamOptimizeImage())
            using (PtlParamOptimizeImageColor paramOptimizeImageColor =
            paramOptimizeImage.getParamOptimizeImageColor())
            {


                // 最適化する画像の対象Filterを設定
                // それぞれの論理和でON・OFFを指定する。
                int filterFlag = (int)PtlParamOptimizeImage.FILTER_TYPE.FILTER_NONE;

                if (validFilter[0])
                {
                    filterFlag = filterFlag | (int)PtlParamOptimizeImage.FILTER_TYPE.FILTER_ASCII85Decode;
                }
                if (validFilter[1])
                {
                    filterFlag = filterFlag | (int)PtlParamOptimizeImage.FILTER_TYPE.FILTER_ASCIIHexDecode;
                }
                if (validFilter[2])
                {
                    filterFlag = filterFlag | (int)PtlParamOptimizeImage.FILTER_TYPE.FILTER_CCITTFaxDecode;
                }
                if (validFilter[3])
                {
                    filterFlag = filterFlag | (int)PtlParamOptimizeImage.FILTER_TYPE.FILTER_DCTDecode;
                }
                if (validFilter[4])
                {
                    filterFlag = filterFlag | (int)PtlParamOptimizeImage.FILTER_TYPE.FILTER_FlateDecode;
                }
                if (validFilter[5])
                {
                    filterFlag = filterFlag | (int)PtlParamOptimizeImage.FILTER_TYPE.FILTER_JBIG2Decode;
                }
                if (validFilter[6])
                {
                    filterFlag = filterFlag | (int)PtlParamOptimizeImage.FILTER_TYPE.FILTER_JPXDecode;
                }
                if (validFilter[7])
                {
                    filterFlag = filterFlag | (int)PtlParamOptimizeImage.FILTER_TYPE.FILTER_LZWDecode;
                }
                if (validFilter[8])
                {
                    filterFlag = filterFlag | (int)PtlParamOptimizeImage.FILTER_TYPE.FILTER_RunLengthDecode;
                }
                paramOptimizeImage.setValidFilter((PtlParamOptimizeImage.FILTER_TYPE)Enum.ToObject(typeof(PtlParamOptimizeImage.FILTER_TYPE), filterFlag));


                // 画像のダウンサンプリングを行う最低サンプル数を設定
                paramOptimizeImage.setMinSampleSize(110);

                // その他のダウンサンプリングの値を設定
                paramOptimizeImageColor.setSourcePPI(150);
                paramOptimizeImageColor.setTargetPPI(75);
                paramOptimizeImageColor.setDownSampling(PtlParamOptimizeImageDownSampling.DOWNSAMPLING_TYPE.DOWNSAMPLING_BICUBIC);
                paramOptimizeImageColor.setMinDownSamplingRate(0.65f);
                paramOptimizeImageColor.setCompress(PtlParamOptimizeImageColor.COMPRESS_TYPE.COMPRESS_AUTO);
                paramOptimizeImageColor.setQuality(PtlParamOptimizeImageColor.QUALITY_TYPE.QUALITY_MIDDLE);


                // 最適化の実行
                doc.optimize(paramOptimize);
            }
        }

        /**
         * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
         * 
         * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
         * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
         * @return argsの数値を読み取った結果を戻す
         * @throws lArgumentException argsが0か1でなかった場合に発生。
         */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;

            // argsの読み取り
            switch (args)
            {
                case "0":
                    trueOrFalse = false;
                    break;
                case "1":
                    trueOrFalse = true;
                    break;
                default:
                    throw new ArgumentException(errorMessage);
            }

            return trueOrFalse;
        }
    }
}
