﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：指定矩形内の文字が削除されます。指定矩形の線上の文字も削除対象です。
  　　　

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class RemoveImageInMask
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: RemoveImageInMask in-pdf-file out-pdf-file" +
                               " page-num");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                printUsage();
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロード
                    doc.load(inputFile);

                    // コマンドライン引数の判定
                    int pageToEdit = int.Parse(args[2]);
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);
                    if ((numPages < 0) || (numPages < pageToEdit))
                    {
                        Console.WriteLine("page-numは入力PDFの全ページ数よりも小さい正の値を" +
                                           "指定してください。");
                        return;
                    }

                    using (PtlPages pages = doc.getPages())
                    {// ページコンテナの取得
                     // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空\n");
                            return;
                        }

                        // ページの取得(パラメータindexは0が先頭のため1を引く)
                        using (PtlPage page = pages.get(pageToEdit - 1))
                        using (PtlParamSetMask paramSetMask = new PtlParamSetMask())
                        // 矩形の形を指定
                        using (PtlRect rect = new PtlRect(setRectCoordinate()))
                        {
                            // 削除対象を画像のみに設定
                            paramSetMask.setRemoveElement(PtlParamSetMask.REMOVE_ELEMENT.REMOVE_IMAGE);

                            // マスク設定
                            paramSetMask.appendRect(rect);
                            paramSetMask.setOpacity(0.3f);
                            page.setMask(paramSetMask);
                        }
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        /**
         * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         * 特に外部からの呼び出しを想定しないためprivateとする。
         * 
         * @param br BufferedReader。数値の読み取りに使う。
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         */
        private static PtlRect setRectCoordinate()
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            PtlRect outputRect = new PtlRect();
            while (!isValueOkay)
            {
                Console.WriteLine("配置矩形の各数値を入力してください。");
                Console.Write("top (mm) : ");
                top = float.Parse(Console.ReadLine());
                Console.Write("bottom (mm) : ");
                bottom = float.Parse(Console.ReadLine());
                if (top < bottom)
                {
                    Console.WriteLine("topの値はbottomよりも大きい値を指定して" +
                                       "再度入力してください。");
                    continue;
                }
                Console.Write("left (mm) : ");
                left = float.Parse(Console.ReadLine());
                Console.Write("right (mm) : ");
                right = float.Parse(Console.ReadLine());
                if (right < left)
                {
                    Console.WriteLine("rightの値はleftよりも大きい値を指定して" +
                                       "再度入力してください。");
                    continue;
                }

                //矩形を正しく指定できた場合の処理
                isValueOkay = true;
                outputRect.setLeft(left);
                outputRect.setBottom(bottom);
                outputRect.setRight(right);
                outputRect.setTop(top);
            }
            return outputRect;
        }
    }
}
