﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：与えられたキーワードで検索し、ヒットした単語の矩形を表示する。
        大文字・小文字の区別、ActualTextを無視するか否かのオプションを指定する。
        0ページを指定した場合はドキュメント全体から検索。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class SearchTextSetTextOption
    {
        static readonly int NUM_OF_ARG_BEFORE_TEXT = 4;

        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: SearchTextSetTextOption in-pdf-file" +
                               " page-to-search compare-case ignore-ActualText" +
                               " searching-text... ");
            Console.WriteLine("page-to-search:");
            Console.WriteLine("0: 全ページから検索 0以外: 指定ページから検索");
            Console.WriteLine("compare-case : ");
            Console.WriteLine("0:大文字と小文字を区別しない 1:区別する");
            Console.WriteLine("ignore-ActualText");
            Console.WriteLine("0:ActualTextを無視しない 1:無視する");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 5)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の読み取り・判定
            int pageToSearch = int.Parse(args[1]); // ページ数
            bool compareCase = true;
            try
            {
                compareCase = readBoolArgs(args[2], "compare-caseは 1か0で指定してください。");
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }
            bool ignoreActualText = true;
            try
            {
                ignoreActualText =
                    readBoolArgs(args[3], "igonre-ActualTextは 1か0で指定してください。");
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }
            // 検索テキストの設定
            int numOfSearchingText = args.Length - NUM_OF_ARG_BEFORE_TEXT;
            String[] arrayTextsToSearch = new String[numOfSearchingText];
            for (int i = 0; i < numOfSearchingText; i++)
            {
                arrayTextsToSearch[i] = args[i + NUM_OF_ARG_BEFORE_TEXT];
            }


            try 
        {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロード
                    doc.load(inputFile);

                    // ページ番号が0を指定された場合はドキュメント全体から検索。
                    if (pageToSearch == 0)
                    {
                        searchTextInDocument(doc, arrayTextsToSearch, compareCase,
                                             ignoreActualText);
                    }
                    else
                    {
                        // それ以外であれば指定ページ番号を検索。
                        int wholePageNum = doc.getPageCount();
                        //pageToSearchのエラー処理
                        if (wholePageNum < pageToSearch)
                        {
                            Console.WriteLine("ERROR: page-to-searchはPDFの総ページ数より" +
                                               "小さい値を指定してください。");
                            Console.WriteLine("総ページ数：" + wholePageNum);
                            printUsage();
                            return;
                        }
                        // 指定ページ番号内を検索。
                        searchTextInPage(doc, arrayTextsToSearch, pageToSearch, compareCase,
                                         ignoreActualText);
                    }
                }
            }
        catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        /**
         * 
         * @param doc
         * @param arrayTextsToSearch 
         */
        private static void searchTextInDocument(PtlPDFDocument doc,
                                                 String[] arrayTextsToSearch,
                                                 bool compareCase, bool ignoreActualText)
        {
            Console.WriteLine("ドキュメント全体から検索します。");
            using (PtlParamSearchText paramSearchText = new PtlParamSearchText())
            {
                // 大文字と小文字を区別するか設定
                paramSearchText.setCompareCase(compareCase);
                // ActualTextを無視するか設定
                paramSearchText.setIgnoreActualText(ignoreActualText);


                // 検索パラメータに対象文字列を追加
                foreach (String textToSearch in arrayTextsToSearch)
                {
                    paramSearchText.appendText(textToSearch);
                }
                // 検索結果コンテナクラスに検索結果を取得し、displaySearchResultsを呼び出す
                using (PtlSearchTextResults resultsContainer = doc.searchText(paramSearchText))
                {
                    displaySearchResults(resultsContainer);
                }
            }

        }

        private static void searchTextInPage(PtlPDFDocument doc,
                                             String[] arrayTextsToSearch,
                                             int pageToSearch,
                                             bool compareCase,
                                             bool ignoreActualText)
        {
            Console.WriteLine(pageToSearch + "ページ目から検索します。");
            //ページコンテナの取得
            using (PtlPages pages = doc.getPages())
            {
                // ページコンテナが空かどうか
                if (pages.isEmpty())
                {
                    Console.WriteLine("ERROR : ページコンテナが空");
                    return;
                }

                // ページの取得(index番号は0が先頭のため1を引く))
                using (PtlPage page = pages.get(pageToSearch - 1))
                using (PtlParamSearchText paramSearchText = new PtlParamSearchText())
                {
                    // 大文字と小文字を区別するか設定
                    paramSearchText.setCompareCase(compareCase);
                    // ActualTextを無視するか設定
                    paramSearchText.setIgnoreActualText(ignoreActualText);

                    // 検索パラメータに対象文字列を追加
                    foreach (String textToSearch in arrayTextsToSearch)
                    {
                        paramSearchText.appendText(textToSearch);
                    }
                    // 検索結果コンテナクラスに検索結果を取得
                    using (PtlSearchTextResults resultsContainer =
                        page.searchText(paramSearchText))
                    {
                        displaySearchResults(resultsContainer);
                    }
                }
            }
        }

        // 検索結果を表示する関数
        private static void displaySearchResults(PtlSearchTextResults resultsContainer)
        {
            // 検索件数の取得
            int resultCount = resultsContainer.getCount();
            Console.WriteLine("検索結果 : " + resultCount + "件見つかりました。");

            // 各検索結果の出力
            for (int i = 0; i < resultCount; i++)
            {
                Console.Write("searchResult No." + (i + 1) + " : ");

                // index番号が i のテキスト検索結果・ページ番号を取得。
                // （index番号、ページ番号は0が先頭）
                using (PtlSearchTextResult searchResult = resultsContainer.get(i))
                {
                    Console.WriteLine((searchResult.getPageNumber() + 1) + "ページ目");
                    Console.WriteLine("searchResult keyword : " + searchResult.getKeyword());
                    // 検索結果の詳細コンテナを取得。
                    using (PtlSearchTextResultDetails resultDetails =
                        searchResult.getResultDetails())
                    {

                        // resultDetailsの中身が1個だった場合（処理の軽量化のため場合分け）
                        if (resultDetails.getCount() == 1)
                        {
                            printResultQuadPoints(resultDetails.get(0));
                        }
                        else
                        {
                            // 複数だった場合
                            for (int j = 0; j < resultDetails.getCount(); j++)
                            {
                                Console.WriteLine("searchResultDetail No. :" + (j + 1));
                                printResultQuadPoints(resultDetails.get(j));
                            }
                        }
                    }
                }
            }
        }

        // 該当テキスト検索結果の詳細コンテナ及びQuadPointを取得して表示する。
        private static void printResultQuadPoints(PtlSearchTextResultDetail resultDetail)
        {
            Console.WriteLine("resultDetail keyword : " + resultDetail.getKeyword());
            // QuadPointの取得
            using (PtlQuadPoint quadPoint = resultDetail.getQuadPoint())
            {

                // 各座標の表示
                Console.WriteLine("該当箇所の矩形座標(x,y)はそれぞれ" +
                                   "左下・右下・左上・右上の順に、");
                printXYCoordinate(quadPoint.getBottomLeft());
                printXYCoordinate(quadPoint.getBottomRight());
                printXYCoordinate(quadPoint.getTopLeft());
                printXYCoordinate(quadPoint.getTopRight());
                Console.WriteLine("です。");
            }
        }

        private static void printXYCoordinate(PtlPoint point)
        {
            Console.WriteLine("(" + point.getX() + "," + point.getY() + "), ");
        }

        /**
         * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
         * 
         * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
         * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
         * @return argsの数値を読み取った結果を戻す
         * @throws ArgumentException argsが0か1でなかった場合に発生。
         */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;

            // argsの読み取り
            switch (args)
            {
                case "0":
                    trueOrFalse = false;
                    break;
                case "1":
                    trueOrFalse = true;
                    break;
                default:
                    throw new ArgumentException(errorMessage);
            }

            return trueOrFalse;
        }
    }
}
