/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：そのPDFに含まれる画像をJPEG圧縮します。
  　　　このとき、

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class CompressImageByJPEG {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java CompressImageByJPEG in-pdf-file out-pdf-file");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 2) {
            printUsage();
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            // 最適化の実行
            compressImageByJPEG(doc);

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    /**
     * そのPDF文書の画像をJPEG圧縮する。
     * @param doc 処理を行うPDFの文書を表現したクラス
     * @throws PtlException PdfToolAPIを実行するときに発生した例外
     */
    public static void compressImageByJPEG(PtlPDFDocument doc) throws PtlException {
        // 画像最適化パラメーターの取得
        // カラー画像最適化パラメーターの取得
        // グレースケール画像最適化パラメーターの取得
        try (PtlParamOptimize paramOptimize = new PtlParamOptimize();
             PtlParamOptimizeImage paramOptimizeImage =
             paramOptimize.getParamOptimizeImage();
             PtlParamOptimizeImageColor paramOptimizeImageColor =
             paramOptimizeImage.getParamOptimizeImageColor();
             PtlParamOptimizeImageGrayScale paramOptimizeImageGrayScale =
             paramOptimizeImage.getParamOptimizeImageGrayScale()) {

            paramOptimizeImageColor.setDownSampling(PtlParamOptimizeImageDownSampling.DOWNSAMPLING_TYPE.DOWNSAMPLING_NONE);
            paramOptimizeImageColor.setCompress(PtlParamOptimizeImageColor.COMPRESS_TYPE.COMPRESS_JPEG);
            paramOptimizeImageColor.setQuality(PtlParamOptimizeImageColor.QUALITY_TYPE.QUALITY_MIDDLE);

            paramOptimizeImageGrayScale.setDownSampling(PtlParamOptimizeImageDownSampling.DOWNSAMPLING_TYPE.DOWNSAMPLING_NONE);
            paramOptimizeImageGrayScale.setCompress(PtlParamOptimizeImageGrayScale.COMPRESS_TYPE.COMPRESS_JPEG);
            paramOptimizeImageGrayScale.setQuality(PtlParamOptimizeImageGrayScale.QUALITY_TYPE.QUALITY_MIDDLE);

            // 最適化する画像の対象Filterを設定
            paramOptimizeImage.setValidFilter(PtlParamOptimizeImage.FILTER_ALL & ~PtlParamOptimizeImage.FILTER_JBIG2Decode);

            //各要素の最適化フラグを設定
            paramOptimize.setRemoveOpenAction(true);
            paramOptimize.setRemoveOutlines(true);
            paramOptimize.setRemoveAnnots(true);
            paramOptimize.setRemoveArticles(true);
            paramOptimize.setRemoveThumbnails(true);

            // 最適化
            doc.optimize(paramOptimize);
        }
    }

}
