/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：レイヤーの描画　指定したページにもう一つのPDFを重ねて表示します。
        重ねたPDFの透明度を数値で設定します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/



package cookbook;

import jp.co.antenna.ptl.*;

public class DrawLayerSetOpacity {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java DrawLayerSetOpacity in-pdf-file out-pdf-file" +
                           " page-num insert-pdf-file page-num-to-insert opacity");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 5) {
            printUsage();
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            // コマンドライン引数の判定
            int pageToAdd = Integer.parseInt(args[2]);
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);
            if((numPages < 0)||(numPages < pageToAdd)) {
                System.err.println("page-numは入力PDFの全ページ数よりも小さい正の値を" +
                                   "指定してください。");
                return;
            }
            String insertPDFPath = args[3];
            int insertPageNum = Integer.parseInt(args[4]);
            float opacity = Float.parseFloat(args[5]);
            if((opacity < 0)||(1<opacity)){
                System.out.println("opacity は0から1の間の小数値を指定してください。");
            }

            try (PtlPages pages = doc.getPages()) { // ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ページコンテナが空");
                    return;
                }

                // ページの取得(パラメータindexは0が先頭のため1を引く)
                try (PtlPage page = pages.get(pageToAdd - 1)) {
                    //レイヤーの描画
                    drawLayerSetOpacity(page, insertPDFPath, insertPageNum, opacity);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void drawLayerSetOpacity(PtlPage page, String insertPDFPath,
                                           int insertPageNum, float opacity)
        throws PtlException, Exception, Error {
        try (PtlSize pageSize = page.getSize();    // ページサイズ
             PtlContent content = page.getContent(); // ページコンテントの取得
             PtlPDFDocument docToInsert = new PtlPDFDocument();
             //挿入するPDF
             PtlParamInput inputFileToInsert = new PtlParamInput(insertPDFPath);
             // 描画矩形の指定（PDFいっぱいに指定）
             PtlRect rect = new PtlRect(0.0f,
                                        0.0f,
                                        pageSize.getWidth(),
                                        pageSize.getHeight())) {
            // PDFファイルをロードします。
            docToInsert.load(inputFileToInsert);

            try (PtlPages pagesToInsert = docToInsert.getPages()) { // ページコンテナの取得
                // ページコンテナが空かどうか
                if (pagesToInsert.isEmpty()) {
                    System.out.println("ページコンテナが空");
                    return;
                }

                // ページの取得(パラメータindexは0が先頭のため1を引く)
                // レイヤーのパラメーターの取得
                try (PtlPage pageInsert = pagesToInsert.get(insertPageNum - 1);
                     PtlParamDrawLayer param = new PtlParamDrawLayer()) {
                    // 透明度を設定
                    param.setOpacity(opacity);

                    // レイヤーにするページ
                    param.setPage(pageInsert);
                    // レイヤーの名前
                    param.setName("Layer1");
                    // レイヤーを前面に
                    param.setZorder(PtlParamDrawLayer.ZORDER.ZORDER_FRONT);
                    // レイヤーを表示
                    param.setShow(PtlParamDrawLayer.SHOW.SHOW_ON);
                    // レイヤーの描画
                    content.drawLayer(rect, PtlContent.ALIGN.ALIGN_CENTER, param);
                }
            }
        }
    }
}
