/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：指定されたインデックス番号の画像エレメントを取得し、
  　　　画像エレメントからファイルを書き出します。
  　　　引数で、出力のフォルダパスとファイル名（ボディ部まで）を指定します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class GetImageInfo {
    
    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java GetImageInfo in-pdf-file" +
                           " page-of-image image-index-num");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage();
            return;
        }

        // コマンドライン引数の取得
        int pageOfImage = Integer.parseInt(args[1]);
        int imageIndexNum = Integer.parseInt(args[2]);

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlPages pages = doc.getPages()) { // ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ページコンテナが空");
                    return;
                }

                // それ以外であれば指定ページ番号を検索。
                int wholePageNum = doc.getPageCount();
                //pageToSearchのエラー処理
                if(wholePageNum < pageOfImage) {
                    System.out.println("ERROR: page-of-image はPDFの総ページ数より"+
                                       "小さい値を指定してください。");
                    System.out.println("総ページ数：" + wholePageNum);
                    printUsage();
                    return;
                }
                // ページの取得(index番号は0が先頭のため1を引く)
                try (PtlPage page = pages.get(pageOfImage - 1)) {
                    System.out.println("このPDFの"+ pageOfImage +"ページ目・Index番号 " +
                                       imageIndexNum + "の画像の情報を読み取ります。");
                    // 画像情報の読み取り
                    getImageInfo(page, imageIndexNum);
                }
            }
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void getImageInfo(PtlPage page, int imageIndexNum)
        throws PtlException, Exception, Error {
        // ページコンテント・画像エレメントの取得
        try (PtlContent content = page.getContent();
             PtlEditElements elems = content.getEditElements(PtlContent.GET_IMAGE)) {
            int numElems = elems.getCount();
            if (numElems == 0) {
                System.out.println("指定されたページに画像エレメントがありませんでした。");
                return; // 画像エレメントが無い場合はreturnする
            }
            //imageIndexNumのエラー処理
            if(numElems < imageIndexNum) {
                System.out.println("ERROR: image-index-numは対象ページの総画像数より"+
                                   "小さい値を指定してください。");
                System.out.println("総画像数：" + numElems);
                printUsage();
                return;
            }

            // OUTPUT_FORMATはFORMAT_AUTOで設定。
            PtlEditImage.OUTPUT_FORMAT format = PtlEditImage.OUTPUT_FORMAT.FORMAT_AUTO;
            // 画像エレメントの取得
            try (PtlEditElement elem = elems.get(imageIndexNum - 1)) {
                PtlEditElement.ELEMENT_TYPE type = elem.getType();
                switch (type) {
                case TYPE_IMAGE:
                    PtlEditImage elemImage = (PtlEditImage)elem;
                    System.out.println("画像の高さ：" + elemImage.getHeight());
                    System.out.println("画像の幅　：" + elemImage.getWidth());
                    System.out.println("画像のPPI ：" + elemImage.getPPI());
                    break;
                default:
                    System.out.println("指定されたエレメントは画像ではありません。");
                    break;
                }
            }
        }
    }
}
