/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：カラー画像に最適化オプションを適用し、PDFを軽量化します。
        最適化をかける最小の画像サイズを指定します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class OptimizeImageSetMinSampleSize {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java OptimizeImageSetMinSampleSize in-pdf-file" +
                           " out-pdf-file min-sample-size");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage();
            return;
        }

        // コマンドライン引数の取得
        int minSampleSize = Integer.parseInt(args[2]);

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            // 画像の最適化を行います。
            optimizeDocSetMinSampleSize(doc, minSampleSize);

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void optimizeDocSetMinSampleSize(PtlPDFDocument doc,
                                                   int minSampleSize)
        throws PtlException{
        try(PtlParamOptimize paramOptimize = new PtlParamOptimize();
            // 画像最適化パラメーターの取得
            // カラー画像最適化パラメーターの取得
            PtlParamOptimizeImage paramOptimizeImage =
            paramOptimize.getParamOptimizeImage();
            PtlParamOptimizeImageColor paramOptimizeImageColor =
            paramOptimizeImage.getParamOptimizeImageColor()) {

            // 画像のダウンサンプリングを行う最低サンプル数を設定
            paramOptimizeImage.setMinSampleSize(minSampleSize);


            // 最適化する画像の対象Filterを設定
            paramOptimizeImage.setValidFilter(PtlParamOptimizeImage.FILTER_ALL & ~PtlParamOptimizeImage.FILTER_JBIG2Decode);

            // その他のダウンサンプリングの値を設定
            paramOptimizeImageColor.setSourcePPI(150);
            paramOptimizeImageColor.setTargetPPI(75);
            paramOptimizeImageColor.setDownSampling(PtlParamOptimizeImageDownSampling.DOWNSAMPLING_TYPE.DOWNSAMPLING_BICUBIC);
            paramOptimizeImageColor.setMinDownSamplingRate(0.65f);
            paramOptimizeImageColor.setCompress(PtlParamOptimizeImageColor.COMPRESS_TYPE.COMPRESS_AUTO);
                paramOptimizeImageColor.setQuality(PtlParamOptimizeImageColor.QUALITY_TYPE.QUALITY_MIDDLE);


            // 最適化の実行
            doc.optimize(paramOptimize);
        }
    }
    
}
