/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：最適化オプションのフォント統合を適用する。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class OptimizeMargeFontInfo {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java OptimizeMargeFontInfo in-pdf-file out-pdf-file" +
                           " set-marge-font");
        System.out.println("set-marge-font : ");
        System.out.println("0:フォント情報を統合しない 1:フォント情報を統合する");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage();
            return;
        }

        boolean setMargeFont = true;
        try {
            setMargeFont = readBoolArgs(args[2], "set-marge-fontは 1か0で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            // 最適化の実行
            optimizeDocSetMargeFontInfo(doc, setMargeFont);

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    /**
     * PDFの最適化を行う。
     * フォントリソースの情報の統合を行うかどうかを選択する。
     * @param doc 処理を行うPDFの文書を表現したクラス
     * @param setMargeFont フォントリソース情報の統合実行についての可否
     * @throws PtlException PdfToolAPIを実行するときに発生した例外
     */
    public static void optimizeDocSetMargeFontInfo(PtlPDFDocument doc, boolean setMargeFont)
        throws PtlException {
        // 最適化パラメーターの取得
        try (PtlParamOptimize paramOptimize = new PtlParamOptimize()) {
            
            //各要素の最適化フラグを設定
            paramOptimize.setMergeFonts(setMargeFont);

            // 最適化
            doc.optimize(paramOptimize);
        }
    }

    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
        throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }
}
