/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：画像を最適化するかどうか、true/falseで指定できる。
        画像の最適化パラメータを指定しない場合はデフォルト値で最適化される。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class OptimizeSetOptimizeImage {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java OptimizeSetOptimizeImage in-pdf-file out-pdf-file" +
                           " set-optimize-image");
        System.out.println("set-optimize-image：");
        System.out.println("0:画像を最適化しない 1:画像を最適化する");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage();
            return;
        }

        // コマンドライン引数の取得
        boolean optimizeImage = true;
        try {
            optimizeImage =
                readBoolArgs(args[2], "set-optimize-image は 0か1で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            // 画像の最適化を行います。
            setOptimizeImageInDoc(doc, optimizeImage);

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void setOptimizeImageInDoc(PtlPDFDocument doc,
                                             boolean optimizeImage)
        throws PtlException{
        try(PtlParamOptimize paramOptimize = new PtlParamOptimize();
            // 画像最適化パラメーターの取得
            // カラー画像最適化パラメーターの取得
            PtlParamOptimizeImage paramOptimizeImage =
            paramOptimize.getParamOptimizeImage();
            PtlParamOptimizeImageColor paramOptimizeImageColor =
            paramOptimizeImage.getParamOptimizeImageColor()) {
            // 画像の最適化を行うかどうか設定。
            paramOptimize.setOptimizeImage(optimizeImage);


            // 最適化する画像の対象Filterを設定
            paramOptimizeImage.setValidFilter(PtlParamOptimizeImage.FILTER_ALL & ~PtlParamOptimizeImage.FILTER_JBIG2Decode);
            // 画像のダウンサンプリングを行う最低サンプル数を設定
            paramOptimizeImage.setMinSampleSize(110);

            // その他のダウンサンプリングの値を設定
            paramOptimizeImageColor.setSourcePPI(150);
            paramOptimizeImageColor.setTargetPPI(75);
            paramOptimizeImageColor.setDownSampling(PtlParamOptimizeImageDownSampling.DOWNSAMPLING_TYPE.DOWNSAMPLING_BICUBIC);
            paramOptimizeImageColor.setMinDownSamplingRate(0.65f);
            paramOptimizeImageColor.setCompress(PtlParamOptimizeImageColor.COMPRESS_TYPE.COMPRESS_AUTO);
                paramOptimizeImageColor.setQuality(PtlParamOptimizeImageColor.QUALITY_TYPE.QUALITY_MIDDLE);

            // 最適化の実行
            doc.optimize(paramOptimize);
        }
    }
    

    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
        throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }

}
