/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：与えられたキーワードで検索し、ヒットした単語の矩形をマスクする。
        検索するページを指定可能。0ページを指定した場合はドキュメント全体から検索。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class SearchTextAndSetMask {
    static final int NUM_OF_ARG_BEFORE_TEXT = 3;

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java SearchTextAndSetMask in-pdf-file out-pdf-file" +
                           " page-to-search searching-text... ");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            printUsage(); // usageメッセージの表示
            return;
        }


        // コマンドライン引数の読み取り・判定
        int pageToSearch = Integer.parseInt(args[2]); // ページ数
        // 検索テキストの設定
        int numOfSearchingText = args.length - NUM_OF_ARG_BEFORE_TEXT; 
        String[] arrayTextsToSearch = new String[numOfSearchingText];
        for (int i = 0; i < numOfSearchingText; i++){
            arrayTextsToSearch[i] = args[i + NUM_OF_ARG_BEFORE_TEXT];
        }

        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロード
            doc.load(inputFile);

            // ページ番号が0を指定された場合はドキュメント全体から検索。
            if(pageToSearch == 0){
                searchTextAndMaskInDocument(doc, arrayTextsToSearch);
            } else {
                // それ以外であれば指定ページ番号を検索。
                int wholePageNum = doc.getPageCount();
                //pageToSearchのエラー処理
                if(wholePageNum < pageToSearch)
                {
                    System.out.println("ERROR: page-to-searchはPDFの総ページ数より"+
                                       "小さい値を指定してください。");
                    System.out.println("総ページ数：" + wholePageNum);
                    printUsage();
                    return;
                }
                // 指定ページ番号内を検索。
                searchTextAndMaskInPage(doc, arrayTextsToSearch, pageToSearch);
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
    
    /**
     * 
     * @param doc
     * @param arrayTextsToSearch 
     */
    private static void searchTextAndMaskInDocument(PtlPDFDocument doc,
                                             String[] arrayTextsToSearch)
        throws PtlException {
        System.out.println("ドキュメント全体から検索します。");
        try(PtlParamSearchTextAndSetMask paramSearchTextSetMask
            = new PtlParamSearchTextAndSetMask();
            PtlColorDeviceRGB color = new PtlColorDeviceRGB(1.0f,1.0f,0.0f)) {
            // パラメータに色情報・透明度情報を追加
            paramSearchTextSetMask.setColor(color);
            paramSearchTextSetMask.setOpacity(0.75f);

            // パラメータに対象文字列を追加
            for(String textToSearch: arrayTextsToSearch) {
                System.out.println("検索キーワードを追加：" + textToSearch);
                paramSearchTextSetMask.appendText(textToSearch);
            }
            // 検索した後マスク処理をする。
            int numOfText = doc.searchTextAndDoProcess(paramSearchTextSetMask);
            System.out.println(numOfText + "件のキーワードをマスク処理しました。");
        }
    }
        
    private static void searchTextAndMaskInPage(PtlPDFDocument doc,
                                                String[] arrayTextsToSearch,
                                                int pageToSearch)
        throws PtlException {
        System.out.println( pageToSearch +  "ページ目から検索します。");
        //ページコンテナの取得
        try (PtlPages pages = doc.getPages()) {
            // ページコンテナが空かどうか
            if (pages.isEmpty()) {
                System.out.println("ERROR : ページコンテナが空");
                return;
            }

            // ページの取得(index番号は0が先頭のため1を引く)
            try (PtlPage page = pages.get(pageToSearch - 1);
                 PtlParamSearchTextAndSetMask paramSearchTextSetMask
                 = new PtlParamSearchTextAndSetMask();
                 PtlColorDeviceRGB color = new PtlColorDeviceRGB(1.0f,1.0f,0.0f)) {
                // パラメータに色情報・透明度情報を追加
                paramSearchTextSetMask.setColor(color);
                paramSearchTextSetMask.setOpacity(0.75f);

                // パラメータに対象文字列を追加
                for(String textToSearch: arrayTextsToSearch) {
                    System.out.println("検索キーワードを追加：" + textToSearch);
                    paramSearchTextSetMask.appendText(textToSearch);
                }
                // 検索した後マスク処理をする。
                int numOfText = page.searchTextAndDoProcess(paramSearchTextSetMask);
                System.out.println(numOfText + "件のキーワードをマスク処理しました。");

            }
        }
    }
    
    // 検索結果を表示する関数
    private static void displaySearchResults(PtlSearchTextResults resultsContainer)
        throws PtlException {
        // 検索件数の取得
        int resultCount = resultsContainer.getCount();
        System.out.println("検索結果 : " + resultCount + "件見つかりました。");

        // 各検索結果の出力
        for(int i = 0; i < resultCount; i++) {
            System.out.print("searchResult No." + (i + 1) + " : ");

            // index番号が i のテキスト検索結果・ページ番号を取得。
            // （index番号、ページ番号は0が先頭）
            try(PtlSearchTextResult searchResult = resultsContainer.get(i)) {
                System.out.println((searchResult.getPageNumber() + 1) + "ページ目");
                System.out.println("searchResult keyword : " + searchResult.getKeyword());
                // 検索結果の詳細コンテナを取得。
                try(PtlSearchTextResultDetails resultDetails =
                    searchResult.getResultDetails()) {

                    // resultDetailsの中身が1個だった場合（処理の軽量化のため場合分け）
                    if(resultDetails.getCount() == 1) {
                        printResultQuadPoints(resultDetails.get(0));
                    } else {
                        // 複数だった場合
                        for(int j = 0; j < resultDetails.getCount(); j++) {
                            System.out.println("searchResultDetail No. :" + (j + 1));
                            printResultQuadPoints(resultDetails.get(j));
                        }
                    }
                }
            }
        }
    }

// 該当テキスト検索結果の詳細コンテナ及びQuadPointを取得して表示する。
    private static void printResultQuadPoints(PtlSearchTextResultDetail resultDetail)
        throws PtlException {
        System.out.println("resultDetail keyword : " + resultDetail.getKeyword());
            // QuadPointの取得
        try(PtlQuadPoint quadPoint = resultDetail.getQuadPoint()) {
                    
            // 各座標の表示
            System.out.println("該当箇所の矩形座標(x,y)はそれぞれ" +
                               "左下・右下・左上・右上の順に、");
            printXYCoordinate(quadPoint.getBottomLeft());
            printXYCoordinate(quadPoint.getBottomRight());
            printXYCoordinate(quadPoint.getTopLeft());
            printXYCoordinate(quadPoint.getTopRight());
            System.out.println("です。");
        }
    }

    private static void printXYCoordinate(PtlPoint point) throws PtlException {
        System.out.println("(" + point.getX() + "," + point.getY() + "), ");
    }

}
