/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：PDFにマスクを設定します。
  　　　指定したページの矩形を指定してマスクし、その部分に透明度を指定して色を塗ります
        透明にしたことが分かりやすいようにマスクの下のエレメントは削除しません。
        

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;

import jp.co.antenna.ptl.*;

public class SetMaskColorWithOpacity {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
            System.out.println("usage: java SetMaskColor in-pdf-file out-pdf-file" +
                               " page-num colorR colorG colorB opacity");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 7) {
            printUsage();
            return;
        }

        // コマンドライン引数の読み取り
        float colorR = Float.parseFloat(args[3]);
        float colorG = Float.parseFloat(args[4]);
        float colorB = Float.parseFloat(args[5]);
        float opacity = Float.parseFloat(args[6]);
        if((opacity < 0) || (1 < opacity)) {
            System.out.println("opacity は0から1の間の小数値を指定してください。");
            printUsage();
            return;
        }
        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロード
            doc.load(inputFile);
            
            // コマンドライン引数の判定
            int pageToAdd = Integer.parseInt(args[2]);
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);
            if((numPages < 0)||(numPages < pageToAdd)) {
                System.err.println("page-numは入力PDFの全ページ数よりも小さい正の値を" +
                                   "指定してください。");
                return;
            }

            try (PtlPages pages = doc.getPages()) {// ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ページコンテナが空\n");
                    return;
                }

                // ページの取得(パラメータindexは0が先頭のため1を引く)
                try (PtlPage page = pages.get(pageToAdd - 1);
                     PtlParamSetMask paramSetMask = new PtlParamSetMask();
                     // 矩形の形を指定
                     PtlRect rect = new PtlRect();
                     // 矩形の色を指定
                     PtlColorDeviceRGB color = new PtlColorDeviceRGB(colorR,
                                                                     colorG,
                                                                     colorB)) {
                    // マスク色の不透明度を設定
                    paramSetMask.setOpacity(opacity);

                    // マスク設定
                    paramSetMask.appendRect(setRectCoordinate(br, rect));
                    paramSetMask.setMaskColor(color);
                    paramSetMask.setRemoveElement(PtlParamSetMask.REMOVE_NONE);
                    page.setMask(paramSetMask);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    /**
     * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
     * 原点はPDFの左下端。
     * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * @param br BufferedReader。数値の読み取りに使う。
     * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
     * @return 指定したleft, bottom, right, topの数値を持つPtlRect
     * @throws IOException
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    public static PtlRect setRectCoordinate(BufferedReader br, PtlRect outputRect)
           throws IOException, PtlException, Exception, Error {
        float top, bottom, left, right;
        boolean isValueOkay = false;
        
        while(!isValueOkay) {
            System.out.println("指定する矩形の各数値を入力してください。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) {//不正矩形は再入力させる
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            if(right < left) {//不正矩形は再入力させる
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setLeft(left);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setTop(top);
        }
        return outputRect;
    }

}
