﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：ハイライト注釈の作成
  　　　指定した矩形にハイライト注釈を設定する。


  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class AppendAnnotHighlight
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: AppendAnnotHighlight in-pdf-file out-pdf-file" +
                               " opacity set-vertical color-R color-G color-B");
            Console.WriteLine("set-vertical:テキストを縦書き方向に囲むかどうか");
            Console.WriteLine("0 : 横書き方向に囲む 1 : 縦書き方向に囲む");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 2)
            {
                printUsage();
                return;
            }

            // コマンドライン引数の読み取り
            float opacity = float.Parse(args[2]);
            bool setVertical;
            // setVerticalの判定
            try
            {
                setVertical = readBoolArgs(args[3], "set-verticalには" +
                                                   "0か1の数値を指定してください。");
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }
            float colorR = float.Parse(args[4]);
            float colorG = float.Parse(args[5]);
            float colorB = float.Parse(args[6]);

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    using (PtlPages pages = doc.getPages()) // ページコンテナの取得
                    {
                     // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空\n");
                            return;
                        }

                        using (PtlPage page = pages.get(0)) // 1ページ目の取得
                        {
                         // テキスト注釈追加
                            addAnnotHighlight(page, opacity, setVertical,
                                              colorR, colorG, colorB);
                        }
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void addAnnotHighlight(PtlPage page, float opacity,
                                             bool setVertical, float colorR, float colorG,
                                             float colorB)
        {
            using (PtlAnnots annots = page.getAnnots()) // 注釈コンテナの取得
            using (PtlAnnotHighlight annotHighlight = new PtlAnnotHighlight())
            // ハイライト注釈用のQuadPointコンテナを取得
            using (PtlQuadPoints annotQuadPoints = annotHighlight.getQuadPoints())
            using (PtlAnnotPopup annotpopup = new PtlAnnotPopup())
            {// ポップアップ注釈

                //ハイライト注釈用のQuadPointコンテナにQuadPointを設定する。

                float[] coordinate = setFloatCoordinate(); //top, bottom, right, left

                using (PtlQuadPoint outputQuadPoint = new PtlQuadPoint())
                using (PtlPoint topRight = new PtlPoint(coordinate[2], coordinate[0]))
                using (PtlPoint topLeft = new PtlPoint(coordinate[3], coordinate[0]))
                using (PtlPoint bottomRight = new PtlPoint(coordinate[2], coordinate[1]))
                using (PtlPoint bottomLeft = new PtlPoint(coordinate[3], coordinate[1]))
                {
                    outputQuadPoint.setTopRight(topRight);
                    outputQuadPoint.setTopLeft(topLeft);
                    outputQuadPoint.setBottomRight(bottomRight);
                    outputQuadPoint.setBottomLeft(bottomLeft);
                    annotQuadPoints.append(outputQuadPoint);
                }

                // 不透明度を設定 0.0 ～ 1.0。0.0が透明、1.0が不透明
                annotHighlight.setMarkUpCA(opacity);

                // Highlight注釈を縦書きにするか。trueが縦書き、falseが横書き（デフォルト）
                annotHighlight.setVerticalDirection(setVertical);

                // 色を設定 setColor(const PtlColorDeviceRGB& color);
                using (PtlColorDeviceRGB color = new PtlColorDeviceRGB(colorR, colorG, colorB))
                {
                    annotHighlight.setColor(color);
                }

                // 内容を設定（注釈に表示されるテキスト）
                annotHighlight.setTextContents("ハイライト注釈。AppendAnnotHighlightで設定");

                // 日時の設定(2013/01/01 00:00:00)
                using (PtlDate date = new PtlDate(2013, 1, 1, 0, 0, 0))
                {
                    annotHighlight.setDate(date);
                }

                // 注釈フラグを設定（論理和） FLAG_NOROTATE = 0x00000010
                /* 注釈の外観をページにあわせて回転させません */
                annotHighlight.setAnnotFlags(PtlAnnotHighlight.ANNOT_FLAGS.FLAG_NOROTATE);


                // 境界線スタイルを設定 BORDER_SOLID = 1, /* 実線(注釈を囲む実線の矩形) */
                annotHighlight.setBorderStyle(PtlAnnotHighlight.BORDER_STYLE.BORDER_SOLID);

                // 境界線幅を設定 BORDER_WIDTH_THIN = 1, /* 細い */
                annotHighlight.setBorderWidth(
                    PtlAnnotHighlight.BORDER_LINE_WIDTH.BORDER_WIDTH_THIN);

                // ポップアップウィンドウのユーザー名
                annotHighlight.setMarkUpTitle("PDF Tool API");

                // ポップアップ注釈の題
                annotHighlight.setMarkUpSubj("ハイライト注釈");

                // 注釈生成日時の設定(2012/12/31 23:59:59)
                using (PtlDate dateMarkup = new PtlDate(2012, 12, 31, 23, 59, 59))
                {
                    annotHighlight.setMarkUpDate(dateMarkup);
                }

                // ポップアップ
                // 矩形座標を設定 座標の単位はmmで原点(0,0)は左下
                using (PtlRect rectPopup = new PtlRect(100.0f, 50.0f, 150.0f, 150.0f))
                {
                    annotpopup.setRect(rectPopup);
                }

                // オープン状態を設定 true=オープン状態、 false=クローズ状態
                annotpopup.setOpenState(true);

                // ポップアップ注釈を設定
                annotHighlight.setAnnotPopUp(annotpopup);

                // 注釈の追加 
                annots.append(annotHighlight);

            }
        }
        /**
         * QuadPointについて、矩形の各値を入力してその座標値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         * 
         * @param br BufferedReader。数値の読み取りに使う。
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         */
        private static float[] setFloatCoordinate()
        {
            float[] coordinate = new float[4]; //top, bottom, right, left
            bool isValueOkay = false;
            while (!isValueOkay)
            {
                Console.WriteLine("配置矩形の各数値を入力してください。");
                Console.Write("top (mm) : ");
                coordinate[0] = float.Parse(Console.ReadLine());
                Console.Write("bottom (mm) : ");
                coordinate[1] = float.Parse(Console.ReadLine());
                if (coordinate[0] < coordinate[1])
                { // top < bottom
                    Console.WriteLine("topの値はbottomよりも大きい値を指定して" +
                                       "再度入力してください。");
                    continue;
                }
                Console.Write("right (mm) : ");
                coordinate[2] = float.Parse(Console.ReadLine());
                Console.Write("left (mm) : ");
                coordinate[3] = float.Parse(Console.ReadLine());
                if (coordinate[2] < coordinate[3])
                { // right < left
                    Console.WriteLine("rightの値はleftよりも大きい値を指定して" +
                                       "再度入力してください。");
                    continue;
                }

                //矩形を正しく指定できた場合の処理
                isValueOkay = true;
            }
            return coordinate;
        }

        /**
         * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
         * 
         * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
         * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定。
         * @return argsの数値を読み取った結果を戻す
         * @throws ArgumentException argsが0か1でなかった場合に発生。
         */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;

            // argsの読み取り
            switch (args)
            {
                case "0":
                    trueOrFalse = false;
                    break;
                case "1":
                    trueOrFalse = true;
                    break;
                default:
                    throw new ArgumentException(errorMessage);
            }

            return trueOrFalse;
        }
    }
}
