﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：リンク注釈の作成
  　　　入力PDFの1ページ目に各種リンク注釈を挿入してPDF形式で出力する。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class AppendAnnotLink
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.Write("usage: AppendAnnotLink in-pdf-file");
            Console.WriteLine(" out-pdf-file アクションの種類");
            Console.WriteLine("アクションの種類");
            Console.WriteLine("0 : GOTOアクションの設定");
            Console.WriteLine("1 : GOTORアクションの設定");
            Console.WriteLine("2 : Launchアクションの設定");
            Console.WriteLine("3 : URIアクションの設定");
        }

        public static void Main(String[] args)
        {
            if (args.Length < 3)
            {
                printUsage();
                return;
            }


            // コマンドライン引数の取得
            int actionKind = int.Parse(args[2]);
            if ((actionKind < 0) || (3 < actionKind))
            {
                Console.Write("アクションの種類 ");
                Console.WriteLine("は0から3の範囲の整数値で指定してください。");
                printUsage();
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    using (PtlPages pages = doc.getPages()) //ページコンテナの取得
                    {
                     // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空\n");
                            return;
                        }

                        using (PtlPage page = pages.get(0)) //1ページ目の取得
                        {
                         // リンク注釈追加
                            addAnnotLink(pages, page, actionKind);
                        }
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void addAnnotLink(PtlPages pages,
                                        PtlPage page, int actionKind)
        {
            using (PtlAnnots annots = page.getAnnots()) //注釈コンテナの取得
            using (PtlAnnotLink annotlink = new PtlAnnotLink())
            {//PDFのリンク注釈
             // 矩形座標を設定
                using (PtlRect rect = setRectCoordinate())
                {
                    annotlink.setRect(rect);
                }
                // 内容を設定
                annotlink.setTextContents("挿入されるリンク注釈");
                // アクションの設定
                setAction(pages, annotlink, actionKind);
                // 注釈の追加
                annots.append(annotlink);
            }
        }

        public static void setAction(PtlPages pages, PtlAnnotLink annotlink, int actionKind)
        {
            switch (actionKind)
            {
                case 0:
                    // GOTOアクションの設定
                    using (PtlActionGoTo acttiongoto = new PtlActionGoTo())
                    using (PtlDestFit destfit = new PtlDestFit())
                    {//宛先
                     // 宛先ページの設定（最終ページに）
                        destfit.setPageNumber(pages.getCount() - 1);
                        // 宛先の設定
                        acttiongoto.setDest(destfit);
                        // アクションの設定
                        annotlink.setAction(acttiongoto);
                    }
                    break;
                case 1:
                    // GOTORアクションの設定
                    using (PtlActionGoToR actiongotor = new PtlActionGoToR())
                    using (PtlDestFit destfit = new PtlDestFit())
                    {//宛先
                     // 宛先ページの設定（最終ページに）
                        destfit.setPageNumber(pages.getCount() - 1);
                        // 宛先の設定
                        actiongotor.setDest(destfit);
                        // ファイル間移動用PDFファイルを設定
                        actiongotor.setFileName("test.pdf");
                        // 新ウィンドウフラグを設定
                        actiongotor.setNewWindowFlag(true);
                        // アクションの設定
                        annotlink.setAction(actiongotor);
                    }
                    break;
                case 2:
                    // Launchアクションの設定
                    using (PtlActionLaunch actionlaunch = new PtlActionLaunch())
                    {
                        // 起動ファイル名を設定
                        actionlaunch.setFileName("test.txt");
                        // 新ウィンドウフラグを設定
                        actionlaunch.setNewWindowFlag(true);
                        // アクションの設定
                        annotlink.setAction(actionlaunch);
                    }
                    break;
                case 3:
                    // URIアクションの設定
                    using (PtlActionURI actionurl = new PtlActionURI())
                    {
                        // URIを設定
                        actionurl.setURI("http://www.antenna.co.jp/");
                        // アクションの設定
                        annotlink.setAction(actionurl);
                    }
                    break;
            }
        }

        /**
         * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         * 
         * @param br BufferedReader。数値の読み取りに使う。
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         */
        public static PtlRect setRectCoordinate()
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            PtlRect outputRect = new PtlRect();
            while (!isValueOkay)
            {
                Console.WriteLine("配置矩形の各数値を入力してください。");
                Console.Write("top (mm) : ");
                top = float.Parse(Console.ReadLine());
                Console.Write("bottom (mm) : ");
                bottom = float.Parse(Console.ReadLine());
                if (top < bottom)
                {
                    Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                    continue;
                }
                Console.Write("left (mm) : ");
                left = float.Parse(Console.ReadLine());
                Console.Write("right (mm) : ");
                right = float.Parse(Console.ReadLine());
                if (right < left)
                {
                    Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                    continue;
                }

                //矩形を正しく指定できた場合の処理
                isValueOkay = true;
                outputRect.setLeft(left);
                outputRect.setBottom(bottom);
                outputRect.setRight(right);
                outputRect.setTop(top);
            }
            return outputRect;
        }
    }
}
