﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：テキスト注釈の作成

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class AppendAnnotText
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: AppendAnnotText in-pdf-file out-pdf-file" +
                               " TextContents UserName Subject");
            Console.WriteLine("TextContents : 注釈内に表示されるテキスト");
            Console.WriteLine("UserName : 注釈を追加したユーザ名");
            Console.WriteLine("Subject : 注釈の内容を説明する短い主題");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 5)
            {
                printUsage();
                return;
            }

            String textContents = args[2];
            String userName = args[3];
            String subject = args[4];
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    using (PtlPages pages = doc.getPages()) // ページコンテナの取得
                    {
                     // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空\n");
                            return;
                        }
                        using (PtlPage page = pages.get(0)) // 1ページ目の取得
                        {
                         // テキスト注釈追加
                            addAnnotText(page, textContents, userName, subject);
                        }
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void addAnnotText(PtlPage page, String textContents, String userName,
                                        String subject)
        {
            using (PtlAnnots annots = page.getAnnots()) // 注釈コンテナの取得
            using (PtlAnnotText annottext = new PtlAnnotText()) // PDFのテキスト注釈
            using (PtlAnnotPopup annotpopup = new PtlAnnotPopup())  // ポップアップ注釈
            {
             // アイコンタイプ設定 ICON_COMMENT = 1, /* コメント */
                annottext.setIconType(PtlAnnotText.ICON_TYPE.ICON_COMMENT);

                // 矩形座標を設定 座標の単位はmmで原点(0,0)は左下
                using (PtlRect rectSize = annottext.getRect())
                using (PtlRect rectAnnot = new PtlRect(50.0f, 50.0f, 50.0f + rectSize.getRight(),
                                                50.0f + rectSize.getTop()))
                {
                    annottext.setRect(rectAnnot);
                }

                // 内容を設定（注釈用に表示されるテキスト・可読な形式での注釈コンテンツの代替説明）
                annottext.setTextContents(textContents);

                // 日時の設定(2013/01/01 00:00:00)
                using (PtlDate date = new PtlDate(2013, 1, 1, 0, 0, 0))
                {
                    annottext.setDate(date);
                }

                // 注釈フラグを設定（論理和） FLAG_NOROTATE = 0x00000010,
                /* 注釈の外観をページにあわせて回転しません。 */
                annottext.setAnnotFlags(PtlAnnotText.ANNOT_FLAGS.FLAG_NOROTATE);

                // 色を設定 setColor(const PtlColorDeviceRGB& color);
                using (PtlColorDeviceRGB color = new PtlColorDeviceRGB(0.0f, 0.0f, 1.0f))
                {
                    annottext.setColor(color);
                }

                // 境界線スタイルを設定 BORDER_SOLID = 1, /* 実線(注釈を囲む実線の矩形) */
                annottext.setBorderStyle(PtlAnnotText.BORDER_STYLE.BORDER_SOLID);

                // 境界線幅を設定 BORDER_WIDTH_THIN = 1, /* 細い */
                annottext.setBorderWidth(PtlAnnotText.BORDER_LINE_WIDTH.BORDER_WIDTH_THIN);

                // ポップアップウィンドウのタイトル文字列設定
                annottext.setMarkUpTitle(userName);
                // サブジェクトの短い説明設定
                annottext.setMarkUpSubj(subject);

                // 注釈生成日時の設定(2012/12/31 23:59:59)
                using (PtlDate dateMarkup = new PtlDate(2012, 12, 31, 23, 59, 59))
                {
                    annottext.setMarkUpDate(dateMarkup);
                }

                // 不透明度を設定 0.0 ～ 1.0。0.0が透明、1.0が不透明
                annottext.setMarkUpCA(0.8f);

                // ポップアップ
                // 矩形座標を設定 座標の単位はmmで原点(0,0)は左下
                using (PtlRect rectPopup = new PtlRect(100.0f, 50.0f, 150.0f, 150.0f))
                {
                    annotpopup.setRect(rectPopup);
                }

                // オープン状態を設定 true=オープン状態、 false=クローズ状態
                annotpopup.setOpenState(true);

                // ポップアップ注釈を設定
                annottext.setAnnotPopUp(annotpopup);

                // 注釈の追加
                annots.append(annottext);
            }
        }

    }
}
