﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：注釈の種別を判別し、各種固有情報を取得する

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class DistinctAnnotsAndGetInfo
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: DistinctAnnotsAndGetInfo in-pdf-file" +
                               " [out-attach-file]");
        }
        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 1)
            {
                printUsage();
                return;
            }
            String outpathAttach = "";
            if (args.Length == 2)
            {
                outpathAttach = args[1];
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    using (PtlPages pages = doc.getPages()) //ページコンテナの取得
                    {
                      // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空\n");
                            return;
                        }

                        int numPages = pages.getCount();
                        for (int i = 0; i < numPages; i++)
                        {
                            Console.WriteLine("ページ" + (i + 1));

                            using (PtlPage page = pages.get(i))
                            {
                                // 注釈コンテナが空かどうか
                                if (!page.hasAnnots())
                                {
                                    Console.WriteLine("注釈なし");
                                    continue;
                                }

                                using (PtlAnnots annots = page.getAnnots())
                                {
                                    // 注釈数の取得
                                    int numAnnots = annots.getCount();
                                    Console.WriteLine("注釈数 : " + numAnnots);
                                    for (int j = 0; j < numAnnots; j++)
                                    {
                                        Console.WriteLine("index num : " + j +
                                                           "の注釈情報を表示します");
                                        using (PtlAnnot annot = annots.get(j))
                                        {
                                            distinctAnnot(annot, outpathAttach);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void distinctAnnot(PtlAnnot annot, String outpathAttach)
        {
            PtlAnnot.ANNOT_TYPE annotType = annot.getType();
            Console.WriteLine("注釈のタイプを判別し、" +
                               "必要に応じてそのタイプ特有の情報を表示します。 :");
            switch (annotType)
            {
                case PtlAnnot.ANNOT_TYPE.TYPE_TEXT:
                    showAnnotText(annot);
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_LINK:
                    showAnnotLink(annot);
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_STAMP:
                    showAnnotStamp(annot);
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_FILE_ATTACHMENT:
                    showAnnotFileAttachment(annot, outpathAttach);
                    break;
                default:
                    Console.WriteLine("その他の注釈のタイプです。" +
                                       "注釈のタイプ名と注釈全般の情報を表示します。");
                    showAnnotOther(annot);
                    break;
            }
        }

        public static void showAnnotText(PtlAnnot annot)
        {
            Console.WriteLine("ANNOT_TYPE = Text ");
            PtlAnnotText annotText = (PtlAnnotText)annot;
            showAnnotCommon(annotText);
            PtlAnnotText.ICON_TYPE iconType = annotText.getIconType();
            switch (iconType)
            {
                case PtlAnnotText.ICON_TYPE.ICON_COMMENT:
                    Console.WriteLine("ICON_TYPE = COMMENT");
                    break;
                case PtlAnnotText.ICON_TYPE.ICON_HELP:
                    Console.WriteLine("ICON_TYPE = HELP");
                    break;
                case PtlAnnotText.ICON_TYPE.ICON_INSERT:
                    Console.WriteLine("ICON_TYPE = INSERT");
                    break;
                case PtlAnnotText.ICON_TYPE.ICON_KEY:
                    Console.WriteLine("ICON_TYPE = KEY");
                    break;
                case PtlAnnotText.ICON_TYPE.ICON_NEWPARAGRAPH:
                    Console.WriteLine("ICON_TYPE = NEWPARAGRAPH");
                    break;
                case PtlAnnotText.ICON_TYPE.ICON_NOTE:
                    Console.WriteLine("ICON_TYPE = NOTE");
                    break;
                case PtlAnnotText.ICON_TYPE.ICON_PARAGRAPH:
                    Console.WriteLine("ICON_TYPE = PARAGRAPH");
                    break;
                case PtlAnnotText.ICON_TYPE.ICON_CUSTOM:
                    Console.WriteLine("ICON_TYPE = CUSTOM");
                    break;
                case PtlAnnotText.ICON_TYPE.ICON_UNKNOWN:
                    Console.WriteLine("ICON_TYPE = UNKNOWN");
                    break;
            }
            Console.WriteLine("ICON_NAME = \"" + annotText.getIconName() + "\"");
            showAnnotMarkup(annotText);
        }

        public static void showAnnotLink(PtlAnnot annot)
        {
            Console.WriteLine("ANNOT_TYPE = Link ");
            PtlAnnotLink annotLink = (PtlAnnotLink)annot;
            showAnnotCommon(annotLink);
            PtlAnnotLink.HIGHLIGHTING_MODE highlight = annotLink.getHighlightMode();
            switch (highlight)
            {
                case PtlAnnotLink.HIGHLIGHTING_MODE.HIGHLIGHTING_MODE_NONE:
                    Console.WriteLine("HIGHLIGHTING_MODE = NONE");
                    break;
                case PtlAnnotLink.HIGHLIGHTING_MODE.HIGHLIGHTING_MODE_INVERT:
                    Console.WriteLine("HIGHLIGHTING_MODE = INVERT");
                    break;
                case PtlAnnotLink.HIGHLIGHTING_MODE.HIGHLIGHTING_MODE_OUTLINE:
                    Console.WriteLine("HIGHLIGHTING_MODE = OUTLINE");
                    break;
                case PtlAnnotLink.HIGHLIGHTING_MODE.HIGHLIGHTING_MODE_PUSH:
                    Console.WriteLine("HIGHLIGHTING_MODE = PUSH");
                    break;
            }
            using (PtlDest dest = annotLink.getDest())
            {
                showDest(dest);
            }
            using (PtlAction act = annotLink.getAction())
            {
                showAction(act);
            }
        }

        public static void showAnnotStamp(PtlAnnot annot)
        {
            Console.WriteLine("ANNOT_TYPE = Stamp");
            PtlAnnotStamp annotStamp = (PtlAnnotStamp)annot;
            showAnnotCommon(annotStamp);
            PtlAnnotStamp.ICON_TYPE iconType = annotStamp.getIconType();
            switch (iconType)
            {
                case PtlAnnotStamp.ICON_TYPE.ICON_APPROVED:
                    Console.WriteLine("ICON_TYPE = APPROVED");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_AS_IS:
                    Console.WriteLine("ICON_TYPE = AS_IS");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_CONFIDENTIAL:
                    Console.WriteLine("ICON_TYPE = CONFIDENTIAL");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_DEPARTMENTAL:
                    Console.WriteLine("ICON_TYPE = DEPARTMENTAL");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_DRAFT:
                    Console.WriteLine("ICON_TYPE = DRAFT");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_EXPERIMENTAL:
                    Console.WriteLine("ICON_TYPE = EXPERIMENTAL");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_EXPIRED:
                    Console.WriteLine("ICON_TYPE = EXPIRED");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_FINAL:
                    Console.WriteLine("ICON_TYPE = FINAL");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_FOR_COMMENT:
                    Console.WriteLine("ICON_TYPE = FOR_COMMENT");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_FOR_PUBLIC_RELEASE:
                    Console.WriteLine("ICON_TYPE = FOR_PUBLIC_RELEASE");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_NOT_APPROVED:
                    Console.WriteLine("ICON_TYPE = NOT_APPROVED");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_NOT_FOR_PUBLIC_RELEASE:
                    Console.WriteLine("ICON_TYPE = NOT_FOR_PUBLIC_RELEASE");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SOLD:
                    Console.WriteLine("ICON_TYPE = SOLD");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_TOP_SECRET:
                    Console.WriteLine("ICON_TYPE = TOP_SECRET");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_APPROVED:
                    Console.WriteLine("ICON_TYPE = SB_APPROVED");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_COMPLETED:
                    Console.WriteLine("ICON_TYPE = SB_COMPLETED");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_CONFIDENTIAL:
                    Console.WriteLine("ICON_TYPE = SB_CONFIDENTIAL");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_DRAFT:
                    Console.WriteLine("ICON_TYPE = SB_DRAFT");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_FINAL:
                    Console.WriteLine("ICON_TYPE = SB_FINAL");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_FOR_COMMENT:
                    Console.WriteLine("ICON_TYPE = SB_FOR_COMMENT");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_FOR_PUBLIC_RELEASE:
                    Console.WriteLine("ICON_TYPE = SB_FOR_PUBLIC_RELEASE");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_INFORMATIONONLY:
                    Console.WriteLine("ICON_TYPE = SB_INFORMATIONONLY");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_NOT_APPROVED:
                    Console.WriteLine("ICON_TYPE = SB_NOT_APPROVED");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_NOT_FOR_PUBLIC_RELEASE:
                    Console.WriteLine("ICON_TYPE = SB_NOT_FOR_PUBLIC_RELEASE");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_PRELIMINARYRESULTS:
                    Console.WriteLine("ICON_TYPE = SB_PRELIMINARYRESULTS");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_SB_VOID:
                    Console.WriteLine("ICON_TYPE = SB_VOID");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_CUSTOM:
                    Console.WriteLine("ICON_TYPE = CUSTOM");
                    break;
                case PtlAnnotStamp.ICON_TYPE.ICON_UNKNOWN:
                    Console.WriteLine("ICON_TYPE = UNKNOWN");
                    break;
            }
            Console.WriteLine("ICON_NAME = \"" + annotStamp.getIconName() + "\"");
            showAnnotMarkup(annotStamp);
        }

        public static void showAnnotFileAttachment(PtlAnnot annot, String outpathAttach)
        {
            Console.WriteLine("ANNOT_TYPE = File Attachment");
            PtlAnnotFileAttachment annotFileAttachment = (PtlAnnotFileAttachment)annot;
            showAnnotCommon(annotFileAttachment);
            PtlAnnotFileAttachment.ICON_TYPE iconType = annotFileAttachment.getIconType();
            switch (iconType)
            {
                case PtlAnnotFileAttachment.ICON_TYPE.ICON_GRAPH:
                    Console.WriteLine("ICON_TYPE = GRAPH");
                    break;
                case PtlAnnotFileAttachment.ICON_TYPE.ICON_PAPERCLIP:
                    Console.WriteLine("ICON_TYPE = PAPERCLIP");
                    break;
                case PtlAnnotFileAttachment.ICON_TYPE.ICON_PUSHPIN:
                    Console.WriteLine("ICON_TYPE = PUSHPIN");
                    break;
                case PtlAnnotFileAttachment.ICON_TYPE.ICON_TAG:
                    Console.WriteLine("ICON_TYPE = TAG");
                    break;
                case PtlAnnotFileAttachment.ICON_TYPE.ICON_CUSTOM:
                    Console.WriteLine("ICON_TYPE = CUSTOM");
                    break;
                case PtlAnnotFileAttachment.ICON_TYPE.ICON_UNKNOWN:
                    Console.WriteLine("ICON_TYPE = UNKNOWN");
                    break;
            }
            Console.WriteLine("ICON_NAME = \"" + annotFileAttachment.getIconName() + "\"");
            Console.WriteLine("FILE_NAME = \"" + annotFileAttachment.getFileName() + "\"");
            if (!String.IsNullOrEmpty(outpathAttach))
            {
                using (PtlParamOutput outputAttach = new PtlParamOutput(outpathAttach))
                {
                    annotFileAttachment.writeFile(outputAttach);
                }
            }
            showAnnotMarkup(annotFileAttachment);
        }

        public static void showAnnotOther(PtlAnnot annot)
        {
            PtlAnnot.ANNOT_TYPE annotType = annot.getType();
            switch (annotType)
            {
                case PtlAnnot.ANNOT_TYPE.TYPE_FREE_TEXT:
                    Console.WriteLine("ANNOT_TYPE = FreeText");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_LINE:
                    Console.WriteLine("ANNOT_TYPE = Line");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_SQUARE:
                    Console.WriteLine("ANNOT_TYPE = Square");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_CIRCLE:
                    Console.WriteLine("ANNOT_TYPE = Circle");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_POLYGON:
                    Console.WriteLine("ANNOT_TYPE = Polygon");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_POLYLINE:
                    Console.WriteLine("ANNOT_TYPE = Polyline");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_HIGHLIGHT:
                    Console.WriteLine("ANNOT_TYPE = Highlight");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_UNDERLINE:
                    Console.WriteLine("ANNOT_TYPE = Underline");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_SQUIGGLY:
                    Console.WriteLine("ANNOT_TYPE = Squiggly");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_STRIKEOUT:
                    Console.WriteLine("ANNOT_TYPE = Strikeout");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_CARET:
                    Console.WriteLine("ANNOT_TYPE = Caret");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_INK:
                    Console.WriteLine("ANNOT_TYPE = Ink");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_SOUND:
                    Console.WriteLine("ANNOT_TYPE = Sound");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_MOVIE:
                    Console.WriteLine("ANNOT_TYPE = Movie");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_WIDGET:
                    Console.WriteLine("ANNOT_TYPE = Widget");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_SCREEN:
                    Console.WriteLine("ANNOT_TYPE = Screen");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_PRINTER_MARK:
                    Console.WriteLine("ANNOT_TYPE = PrinterMark");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_TRAP_NETWORK:
                    Console.WriteLine("ANNOT_TYPE = TrapNetwork");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_WATERMARK:
                    Console.WriteLine("ANNOT_TYPE = Watermark");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_3D:
                    Console.WriteLine("ANNOT_TYPE = 3D");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_REDACT:
                    Console.WriteLine("ANNOT_TYPE = Redact");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_PROJECTION:
                    Console.WriteLine("ANNOT_TYPE = Projection");
                    break;
                case PtlAnnot.ANNOT_TYPE.TYPE_RICHMEDIA:
                    Console.WriteLine("ANNOT_TYPE = Richmedia");
                    break;
                default:
                    Console.WriteLine("ANNOT_TYPE = Unknown");
                    break;
            }
            showAnnotCommon(annot);
            if (annot.isMarkup())
            {
                showAnnotMarkup(annot);
            }
        }

        public static void showPopup(PtlAnnot annot)
        {
            Console.WriteLine("ANNOT_TYPE = Popup");
            PtlAnnotPopup annotPopup = (PtlAnnotPopup)annot;
            showAnnotCommon(annotPopup);
            bool openState = annotPopup.getOpenState();
            if (openState)
                Console.WriteLine("OPEN_STATE == true");
            else
                Console.WriteLine("OPEN_STATE == false");
        }

        /**
         * 注釈全般に共通する情報を表示する。
         * 共通する情報とは、矩形座標、注釈の内容、注釈フラグ、色、内部色、
         * 境界線スタイル、境界線幅です。
         * @param annot 読み取る注釈
         * @throws PtlException PDF Tool APIにより発生した例外
         * @throws Exception その他の例外
         */
        public static void showAnnotCommon(PtlAnnot annot)
        {
            using (PtlRect rect = annot.getRect())
            {
                Console.WriteLine("Rect(Left, Bottom, Right, Top)");
                Console.Write(" = ({0:f6},{1:f6},{2:f6},{3:f6})\n",
                                  rect.getLeft(), rect.getBottom(),
                                      rect.getRight(), rect.getTop());
            }
            Console.WriteLine("TextContents = \"" + annot.getTextContents() + "\"");
            using (PtlDate date = annot.getDate())
            {
                int year = date.getYear();
                int month = date.getMonth();
                int day = date.getDay();
                int hour = date.getHour();
                int min = date.getMin();
                int sec = date.getSec();
                Console.Write("Date : {0:d}/{1:d}/{2:d} {3:d}:{4:d}:{5:d}\n", year, month, day, hour,
                                  min, sec);
            }
            int flags = (int)annot.getAnnotFlags();
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_INVISIBLE) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_INVISIBLE)
                Console.WriteLine("FLAG_INVISIBLE on");
            else
                Console.WriteLine("FLAG_INVISIBLE off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_HIDDEN) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_HIDDEN)
                Console.WriteLine("FLAG_HIDDEN on");
            else
                Console.WriteLine("FLAG_HIDDEN off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_PRINT) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_PRINT)
                Console.WriteLine("FLAG_PRINT on");
            else
                Console.WriteLine("FLAG_PRINT off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOZOOM) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOZOOM)
                Console.WriteLine("FLAG_NOZOOM on");
            else
                Console.WriteLine("FLAG_NOZOOM off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOROTATE) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOROTATE)
                Console.WriteLine("FLAG_NOROTATE on");
            else
                Console.WriteLine("FLAG_NOROTATE off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOVIEW) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOVIEW)
                Console.WriteLine("FLAG_NOVIEW on");
            else
                Console.WriteLine("FLAG_NOVIEW off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_READONLY) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_READONLY)
                Console.WriteLine("FLAG_READONLY on");
            else
                Console.WriteLine("FLAG_READONLY off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_LOCKED) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_LOCKED)
                Console.WriteLine("FLAG_LOCKED on");
            else
                Console.WriteLine("FLAG_LOCKED off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_TOGGLE_NOVIEW) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_TOGGLE_NOVIEW)
                Console.WriteLine("FLAG_TOGGLE_NOVIEW on");
            else
                Console.WriteLine("FLAG_TOGGLE_NOVIEW off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_LOCKED_CONTENTS) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_LOCKED_CONTENTS)
                Console.WriteLine("FLAG_LOCKED_CONTENTS on");
            else
                Console.WriteLine("FLAG_LOCKED_CONTENTS off");

            using (PtlColor color = annot.getColor())
            {
                Console.WriteLine("show color info:");
                showColor(color);
            }
            using (PtlColor colorI = annot.getInteriorColor())
            {
                Console.WriteLine("show interior color info:");
                showColor(colorI);
            }
            PtlAnnot.BORDER_STYLE boderStyle = annot.getBorderStyle();
            switch (boderStyle)
            {
                case PtlAnnot.BORDER_STYLE.BORDER_SOLID:
                    Console.WriteLine("BORDER_STYLE = SOLID");
                    break;
                case PtlAnnot.BORDER_STYLE.BORDER_DASHED:
                    Console.WriteLine("BORDER_STYLE = DASHED");
                    break;
                case PtlAnnot.BORDER_STYLE.BORDER_BEVELED:
                    Console.WriteLine("BORDER_STYLE = BEVELED");
                    break;
                case PtlAnnot.BORDER_STYLE.BORDER_INSET:
                    Console.WriteLine("BORDER_STYLE = INSET");
                    break;
                case PtlAnnot.BORDER_STYLE.BORDER_UNDERLINE:
                    Console.WriteLine("BORDER_STYLE = UNDERLINE");
                    break;
            }
            PtlAnnot.BORDER_LINE_WIDTH boderWidth = annot.getBorderWidth();
            switch (boderWidth)
            {
                case PtlAnnot.BORDER_LINE_WIDTH.BORDER_WIDTH_NONE:
                    Console.WriteLine("BORDER_WIDTH = NONE");
                    break;
                case PtlAnnot.BORDER_LINE_WIDTH.BORDER_WIDTH_THIN:
                    Console.WriteLine("BORDER_WIDTH = THIN");
                    break;
                case PtlAnnot.BORDER_LINE_WIDTH.BORDER_WIDTH_MIDDLE:
                    Console.WriteLine("BORDER_WIDTH = MIDDLE");
                    break;
                case PtlAnnot.BORDER_LINE_WIDTH.BORDER_WIDTH_THICK:
                    Console.WriteLine("BORDER_WIDTH = THICK");
                    break;
            }
        }

        public static void showAnnotMarkup(PtlAnnot annot)
        {
            PtlAnnotMarkup annotMarkup = (PtlAnnotMarkup)annot;
            Console.WriteLine("MarkUpTitle = \"" + annotMarkup.getMarkUpTitle() + "\"");
            Console.WriteLine("MarkUpSubj = \"" + annotMarkup.getMarkUpSubj() + "\"");
            using (PtlDate date = annotMarkup.getMarkUpDate())
            {
                int year = date.getYear();
                int month = date.getMonth();
                int day = date.getDay();
                int hour = date.getHour();
                int min = date.getMin();
                int sec = date.getSec();
                Console.Write("MarkUpDate : {0:d}/{1:d}/{2:d} {3:d}:{4:d}:{5:d}\n", year, month, day, hour,
                                  min, sec);
            }
            Console.Write("MarkUpCA : {0:f6}\n", annotMarkup.getMarkUpCA());
            if (annotMarkup.hasAnnotPopup())
            {
                using (PtlAnnotPopup popup = annotMarkup.getAnnotPopup())
                {
                    Console.WriteLine("ポップアップ注釈が含まれていました。" +
                                       "その情報を表示します。");
                    showPopup(popup);
                }
            }
        }

        public static void showColor(PtlColor color)
        {
            PtlColor.COLOR_TYPE colorType = color.getType();
            switch (colorType)
            {
                case PtlColor.COLOR_TYPE.TYPE_NONE:
                    Console.WriteLine("COLOR_TYPE = None");
                    break;
                case PtlColor.COLOR_TYPE.TYPE_DEVICE_GRAY:
                    PtlColorDeviceGray colorGray = (PtlColorDeviceGray)color;
                    Console.Write("COLOR_TYPE = GRAY({0:f6})\n", colorGray.getGray());
                    break;
                case PtlColor.COLOR_TYPE.TYPE_DEVICE_RGB:
                    PtlColorDeviceRGB colorRGB = (PtlColorDeviceRGB)color;
                    Console.Write("COLOR_TYPE = RGB({0:f6},{1:f6},{2:f6})\n", colorRGB.getR(),
                                                  colorRGB.getG(), colorRGB.getB());
                    break;
                case PtlColor.COLOR_TYPE.TYPE_DEVICE_CMYK:
                    PtlColorDeviceCMYK colorCMYK = (PtlColorDeviceCMYK)color;
                    Console.Write("COLOR_TYPE = CMYK({0:f6},{1:f6},{2:f6},{3:f6})\n", colorCMYK.getC(),
                                                  colorCMYK.getM(), colorCMYK.getY(), colorCMYK.getK());
                    break;
            }
        }

        public static void showAction(PtlAction act)
        {
            PtlAction.ACTION_TYPE actType = act.getType();
            switch (actType)
            {
                case PtlAction.ACTION_TYPE.TYPE_GOTO:
                    Console.WriteLine("ACTION_TYPE = GOTO");
                    PtlActionGoTo actGoTo = (PtlActionGoTo)act;
                    using (PtlDest destGoTo = actGoTo.getDest())
                    {
                        showDest(destGoTo);
                    }
                    break;
                case PtlAction.ACTION_TYPE.TYPE_GOTO_R:
                    Console.WriteLine("ACTION_TYPE = GOTOR");
                    PtlActionGoToR actGoToR = (PtlActionGoToR)act;
                    using (PtlDest destGoToR = actGoToR.getDest())
                    {
                        showDest(destGoToR);
                    }
                    String fileName1 = actGoToR.getFileName();
                    Console.Write("GOTOR FILENAME = {0}\n", fileName1);
                    if (actGoToR.getNewWindowFlag())
                        Console.WriteLine("GOTOR NewWindowFlag on");
                    else
                        Console.WriteLine("GOTOR NewWindowFlag off");
                    break;
                case PtlAction.ACTION_TYPE.TYPE_LAUNCH:
                    Console.WriteLine("ACTION_TYPE = LAUNCH");
                    PtlActionLaunch actLaunch = (PtlActionLaunch)act;
                    String fileName2 = actLaunch.getFileName();
                    Console.Write("LAUNCH FILENAME = {0}\n", fileName2);
                    if (actLaunch.getNewWindowFlag())
                        Console.WriteLine("LAUNCH NewWindowFlag on");
                    else
                        Console.WriteLine("LAUNCH NewWindowFlag off");
                    break;
                case PtlAction.ACTION_TYPE.TYPE_URI:
                    Console.WriteLine("ACTION_TYPE = URI");
                    PtlActionURI actURI = (PtlActionURI)act;
                    String uri = actURI.getURI();
                    Console.Write("URI = {0}\n", uri);
                    break;
                case PtlAction.ACTION_TYPE.TYPE_UNKNOWN:
                    Console.WriteLine("ACTION_TYPE = UNKNOWN");
                    break;
            }
        }

        public static void showDest(PtlDest dest)
        {
            PtlDest.DEST_TYPE destType = dest.getType();
            switch (destType)
            {
                case PtlDest.DEST_TYPE.TYPE_NONE:
                    Console.WriteLine("DEST_TYPE = NONE");
                    break;
                case PtlDest.DEST_TYPE.TYPE_XYZ:
                    Console.WriteLine("DEST_TYPE = XYZ");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT:
                    Console.WriteLine("DEST_TYPE = FIT");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_H:
                    Console.WriteLine("DEST_TYPE = FITH");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_V:
                    Console.WriteLine("DEST_TYPE = FITV");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_R:
                    Console.WriteLine("DEST_TYPE = FITR");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_B:
                    Console.WriteLine("DEST_TYPE = FITB");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_BH:
                    Console.WriteLine("DEST_TYPE = FITBH");
                    break;
                case PtlDest.DEST_TYPE.TYPE_FIT_BV:
                    Console.WriteLine("DEST_TYPE = FITBV");
                    break;
            }
            Console.Write("DEST_PAGE : {0:d}\n", dest.getPageNumber());
            if (dest.isLeftNull())
                Console.WriteLine("DEST_LEFT = null");
            else
                Console.Write("DEST_LEFT : {0:f6}\n", dest.getLeft());
            if (dest.isBottomNull())
                Console.WriteLine("DEST_BOTTOM = null");
            else
                Console.Write("DEST_BOTTOM : {0:f6}\n", dest.getBottom());
            if (dest.isRightNull())
                Console.WriteLine("DEST_RIGHT = null");
            else
                Console.Write("DEST_RIGHT : {0:f6}\n", dest.getRight());
            if (dest.isTopNull())
                Console.WriteLine("DEST_TOP = null");
            else
                Console.Write("DEST_TOP : {0:f6}\n", dest.getTop());
            if (dest.isZoomNull())
                Console.WriteLine("DEST_ZOOM = null");
            else
                Console.Write("DEST_ZOOM : {0:f6}\n", dest.getZoom());
        }

    }
}
