﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：画像ファイルのPDF化(配置矩形の設定)

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class ImageToPdfSetRect
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: ImageToPdfSetRect image-file" +
                               " out-pdf-file image-align");
            Console.WriteLine("image-align : 画像の配置");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 2)
            {
                printUsage();
                return;
            }

            //コマンドライン引数の読み取り
            String AlignType = args[2];
            try
            {
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())  //PDF文書を表現したクラスを新規作成
                using (PtlPages pages = doc.getPages())  //作成したクラスのページコンテナの取得
                {
                    // 画像描画パラメータ
                    // 画像ファイル
                    // 画像ページパラメータ の取得
                    using (PtlParamDrawImage paramDrawImage = new PtlParamDrawImage())
                    using (PtlParamInput inputImage = new PtlParamInput(args[0]))
                    using (PtlParamImagePage paramImagePage = new PtlParamImagePage())
                    {
                        // 画像の配置を設定
                        try
                        {
                            paramImagePage.setAlign(
                                (PtlParamImagePage.ALIGN)Enum.Parse(typeof(PtlParamImagePage.ALIGN), AlignType));
                        }
                        catch (ArgumentException iaex)
                        {
                            //PtlParamImagePage.PAPER_TYPEの指定が誤っていた場合のエラーメッセージ
                            Console.WriteLine(iaex.Message);
                            Console.WriteLine("ERROR : [画像の配置]には" +
                                               "PtlParamImagePage.ALIGNに" +
                                               "含まれる名前を指定してください。");
                            Console.WriteLine(iaex.StackTrace);
                        }
                        //画像の配置矩形を設定
                        paramImagePage.setRect(setRectCoordinate());

                        // 画像描画パラメータに画像ファイルを設定
                        paramDrawImage.setImageStream(inputImage);
                        // 画像ページパラメータに画像描画パラメータを設定
                        paramImagePage.setImage(paramDrawImage);
                        // 画像ページのサイズを画像サイズにあわせる
                        paramImagePage.setPaperType(PtlParamImagePage.PAPER_TYPE.PAPER_A4);
                        // ページコンテナに画像ページパラメータを追加
                        pages.append(paramImagePage);
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        /**
         * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         * 
         * @param br BufferedReader。数値の読み取りに使う。
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         */
        public static PtlRect setRectCoordinate()
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            PtlRect outputRect = new PtlRect();
            while (!isValueOkay)
            {
                Console.WriteLine("配置矩形の各数値を入力してください。");
                Console.Write("top (mm) : ");
                top = float.Parse(Console.ReadLine());
                Console.Write("bottom (mm) : ");
                bottom = float.Parse(Console.ReadLine());
                if (top < bottom)
                {
                    Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                    continue;
                }
                Console.Write("left (mm) : ");
                left = float.Parse(Console.ReadLine());
                Console.Write("right (mm) : ");
                right = float.Parse(Console.ReadLine());
                if (right < left)
                {
                    Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                    continue;
                }

                //矩形を正しく指定できた場合の処理
                isValueOkay = true;
                outputRect.setLeft(left);
                outputRect.setBottom(bottom);
                outputRect.setRight(right);
                outputRect.setTop(top);
            }
            return outputRect;
        }

    }
}
