﻿/*
  Antenna House PDF Tool API V6.0
  .Net Interface sample program

  概要：注釈情報を取得してマークアップ注釈か否かを判別して情報を表示する。
  　　　ポップアップ注釈が含まれていた場合はそれの情報を表示する。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace cookbook
{
    public class ShowAnnotMarkupInfo
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: ShowAnnotMarkupInfo in-pdf-file");
        }

        /**
         * @param args the command line arguments
         */
        public static void Main(String[] args)
        {
            if (args.Length < 1)
            {
                printUsage();
                return;
            }
            String outpathAttach = "";
            if (args.Length == 2)
            {
                outpathAttach = args[1];
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    using (PtlPages pages = doc.getPages()) //ページコンテナの取得
                    {
                      // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空\n");
                            return;
                        }

                        int numPages = pages.getCount();
                        for (int i = 0; i < numPages; i++)
                        {
                            Console.WriteLine("ページ" + (i + 1));

                            using (PtlPage page = pages.get(i))
                            {
                                // 注釈コンテナが空かどうか
                                if (!page.hasAnnots())
                                {
                                    Console.WriteLine("注釈なし");
                                    continue;
                                }

                                using (PtlAnnots annots = page.getAnnots())
                                {
                                    // 注釈数の取得
                                    int numAnnots = annots.getCount();
                                    Console.WriteLine("注釈数 : " + numAnnots);
                                    for (int j = 0; j < numAnnots; j++)
                                    {
                                        Console.WriteLine("index num : " + j +
                                                           "について、マークアップ注釈か否かを" +
                                                           "判定します。");
                                        using (PtlAnnot annot = annots.get(j))
                                        {
                                            if (annot.isMarkup())
                                            {
                                                Console.WriteLine("マークアップ注釈です。" +
                                                                   "情報を表示します。");
                                                showAnnotMarkup(annot);
                                            }
                                            else
                                            {
                                                Console.WriteLine("マークアップ注釈では" +
                                                                   "ありません。");
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void showAnnotMarkup(PtlAnnot annot)
        {
            PtlAnnotMarkup annotMarkup = (PtlAnnotMarkup)annot;
            Console.WriteLine("MarkUpTitle = \"" + annotMarkup.getMarkUpTitle() + "\"");
            Console.WriteLine("MarkUpSubj = \"" + annotMarkup.getMarkUpSubj() + "\"");
            using (PtlDate date = annotMarkup.getMarkUpDate())
            {
                int year = date.getYear();
                int month = date.getMonth();
                int day = date.getDay();
                int hour = date.getHour();
                int min = date.getMin();
                int sec = date.getSec();
                Console.Write("MarkUpDate : {0:d}/{1:d}/{2:d} {3:d}:{4:d}:{5:d}\n", year, month, day, hour,
                                  min, sec);
            }
            Console.Write("MarkUpCA : {0:f6}\n", annotMarkup.getMarkUpCA());
            if (annotMarkup.hasAnnotPopup())
            {
                using (PtlAnnotPopup popup = annotMarkup.getAnnotPopup())
                {
                    Console.WriteLine("ポップアップ注釈が含まれていました。" +
                                       "その情報を表示します。");
                    showPopup(popup);
                }
            }
        }

        public static void showPopup(PtlAnnot annot)
        {
            Console.WriteLine("ANNOT_TYPE = Popup");
            PtlAnnotPopup annotPopup = (PtlAnnotPopup)annot;
            showAnnotCommon(annotPopup);
            bool openState = annotPopup.getOpenState();
            if (openState)
                Console.WriteLine("OPEN_STATE == true");
            else
                Console.WriteLine("OPEN_STATE == false");
        }

        public static void showAnnotCommon(PtlAnnot annot)
        {
            using (PtlRect rect = annot.getRect())
            {
                Console.WriteLine("Rect(Left, Bottom, Right, Top)");
                Console.Write(" = ({0:f6},{1:f6},{2:f6},{3:f6})\n",
                                  rect.getLeft(), rect.getBottom(),
                                      rect.getRight(), rect.getTop());
            }
            Console.WriteLine("TextContents = \"" + annot.getTextContents() + "\"");
            using (PtlDate date = annot.getDate())
            {
                int year = date.getYear();
                int month = date.getMonth();
                int day = date.getDay();
                int hour = date.getHour();
                int min = date.getMin();
                int sec = date.getSec();
                Console.Write("Date : {0:d}/{1:d}/{2:d} {3:d}:{4:d}:{5:d}\n", year, month, day, hour,
                                  min, sec);
            }
            int flags = (int)annot.getAnnotFlags();
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_INVISIBLE) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_INVISIBLE)
                Console.WriteLine("FLAG_INVISIBLE on");
            else
                Console.WriteLine("FLAG_INVISIBLE off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_HIDDEN) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_HIDDEN)
                Console.WriteLine("FLAG_HIDDEN on");
            else
                Console.WriteLine("FLAG_HIDDEN off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_PRINT) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_PRINT)
                Console.WriteLine("FLAG_PRINT on");
            else
                Console.WriteLine("FLAG_PRINT off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOZOOM) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOZOOM)
                Console.WriteLine("FLAG_NOZOOM on");
            else
                Console.WriteLine("FLAG_NOZOOM off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOROTATE) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOROTATE)
                Console.WriteLine("FLAG_NOROTATE on");
            else
                Console.WriteLine("FLAG_NOROTATE off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOVIEW) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_NOVIEW)
                Console.WriteLine("FLAG_NOVIEW on");
            else
                Console.WriteLine("FLAG_NOVIEW off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_READONLY) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_READONLY)
                Console.WriteLine("FLAG_READONLY on");
            else
                Console.WriteLine("FLAG_READONLY off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_LOCKED) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_LOCKED)
                Console.WriteLine("FLAG_LOCKED on");
            else
                Console.WriteLine("FLAG_LOCKED off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_TOGGLE_NOVIEW) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_TOGGLE_NOVIEW)
                Console.WriteLine("FLAG_TOGGLE_NOVIEW on");
            else
                Console.WriteLine("FLAG_TOGGLE_NOVIEW off");
            if ((flags & (int)PtlAnnot.ANNOT_FLAGS.FLAG_LOCKED_CONTENTS) == (int)PtlAnnot.ANNOT_FLAGS.FLAG_LOCKED_CONTENTS)
                Console.WriteLine("FLAG_LOCKED_CONTENTS on");
            else
                Console.WriteLine("FLAG_LOCKED_CONTENTS off");

            using (PtlColor color = annot.getColor())
            {
                Console.WriteLine("show color info:");
                showColor(color);
            }
            using (PtlColor colorI = annot.getInteriorColor())
            {
                Console.WriteLine("show interior color info:");
                showColor(colorI);
            }
            PtlAnnot.BORDER_STYLE boderStyle = annot.getBorderStyle();
            switch (boderStyle)
            {
                case PtlAnnot.BORDER_STYLE.BORDER_SOLID:
                    Console.WriteLine("BORDER_STYLE = SOLID");
                    break;
                case PtlAnnot.BORDER_STYLE.BORDER_DASHED:
                    Console.WriteLine("BORDER_STYLE = DASHED");
                    break;
                case PtlAnnot.BORDER_STYLE.BORDER_BEVELED:
                    Console.WriteLine("BORDER_STYLE = BEVELED");
                    break;
                case PtlAnnot.BORDER_STYLE.BORDER_INSET:
                    Console.WriteLine("BORDER_STYLE = INSET");
                    break;
                case PtlAnnot.BORDER_STYLE.BORDER_UNDERLINE:
                    Console.WriteLine("BORDER_STYLE = UNDERLINE");
                    break;
            }
            PtlAnnot.BORDER_LINE_WIDTH boderWidth = annot.getBorderWidth();
            switch (boderWidth)
            {
                case PtlAnnot.BORDER_LINE_WIDTH.BORDER_WIDTH_NONE:
                    Console.WriteLine("BORDER_WIDTH = NONE");
                    break;
                case PtlAnnot.BORDER_LINE_WIDTH.BORDER_WIDTH_THIN:
                    Console.WriteLine("BORDER_WIDTH = THIN");
                    break;
                case PtlAnnot.BORDER_LINE_WIDTH.BORDER_WIDTH_MIDDLE:
                    Console.WriteLine("BORDER_WIDTH = MIDDLE");
                    break;
                case PtlAnnot.BORDER_LINE_WIDTH.BORDER_WIDTH_THICK:
                    Console.WriteLine("BORDER_WIDTH = THICK");
                    break;
            }
        }

        public static void showColor(PtlColor color)
        {
            PtlColor.COLOR_TYPE colorType = color.getType();
            switch (colorType)
            {
                case PtlColor.COLOR_TYPE.TYPE_NONE:
                    Console.WriteLine("COLOR_TYPE = None");
                    break;
                case PtlColor.COLOR_TYPE.TYPE_DEVICE_GRAY:
                    PtlColorDeviceGray colorGray = (PtlColorDeviceGray)color;
                    Console.Write("COLOR_TYPE = GRAY({0:f6})\n", colorGray.getGray());
                    break;
                case PtlColor.COLOR_TYPE.TYPE_DEVICE_RGB:
                    PtlColorDeviceRGB colorRGB = (PtlColorDeviceRGB)color;
                    Console.Write("COLOR_TYPE = RGB({0:f6},{1:f6},{2:f6})\n", colorRGB.getR(),
                                                  colorRGB.getG(), colorRGB.getB());
                    break;
                case PtlColor.COLOR_TYPE.TYPE_DEVICE_CMYK:
                    PtlColorDeviceCMYK colorCMYK = (PtlColorDeviceCMYK)color;
                    Console.Write("COLOR_TYPE = CMYK({0:f6},{1:f6},{2:f6},{3:f6})\n", colorCMYK.getC(),
                                                  colorCMYK.getM(), colorCMYK.getY(), colorCMYK.getK());
                    break;
            }
        }
    }
}
