/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：ファイル添付注釈の作成
  　　　ファイル添付注釈を入力PDFの1ページ目に追加して、
  　　　新しいPDFを出力します。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class AppendAnnotFileAttachment {
    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.print("usage: java AppendAnnotFileAttachment in-pdf-file");
        System.out.println(" out-pdf-file attachment-file");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage();
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlPages pages = doc.getPages()) { //ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ページコンテナが空\n");
                    return;
                }

                try (PtlPage page = pages.get(0)) { //1ページ目の取得
                    // ファイル添付注釈追加
                    addAnnotFileAttachment(page, args[2]);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void addAnnotFileAttachment(PtlPage page, String attachmentfilename)
        throws PtlException, Exception, Error {
        try (PtlAnnots annots = page.getAnnots(); //注釈コンテナの取得
             //PDFのファイル添付注釈
             PtlAnnotFileAttachment annotfileattachment = new PtlAnnotFileAttachment();
             PtlAnnotPopup annotpopup = new PtlAnnotPopup()) {// ポップアップ注釈
            // アイコンタイプ設定 ICON_PUSHPIN = 3, /* プッシュピン */
            annotfileattachment.setIconType(PtlAnnotFileAttachment.ICON_TYPE.ICON_PUSHPIN);

            // 矩形座標を設定 座標の単位はmmで原点(0,0)は左下
            try (PtlRect rectSize = annotfileattachment.getRect();
                 PtlRect rectAnnot = new PtlRect(50.0f, 50.0f, 50.0f+rectSize.getRight(),
                                                 50.0f+rectSize.getTop())) {
                annotfileattachment.setRect(rectAnnot);
            }

            // 内容を設定（注釈用に表示されるテキスト・
            // 可読な形式での注釈コンテンツの代替説明）
            annotfileattachment.setTextContents("注釈用に表示されるテキスト" +
                                                "・可読な形式での注釈コンテンツの代替説明");

            // 日時の設定(2000/5/5 12:00:00)
            try (PtlDate date = new PtlDate(2000, 5, 5, 12, 0, 0))
            {
                annotfileattachment.setDate(date);
            }

            // 色を設定
            try (PtlColorDeviceRGB color =  new PtlColorDeviceRGB(1.0f, 0.0f, 1.0f))
            {
                annotfileattachment.setColor(color);
            }

            // 境界線スタイルを設定 BORDER_SOLID = 1, /* 実線(注釈を囲む実線の矩形) */
            annotfileattachment.setBorderStyle(PtlAnnotText.BORDER_STYLE.BORDER_SOLID);

            // 境界線幅を取得 BORDER_WIDTH_THICK = 3 /* 太い */
            annotfileattachment.setBorderWidth(PtlAnnotText.BORDER_LINE_WIDTH.BORDER_WIDTH_THICK);

            // ポップアップウィンドウのタイトル文字列設定
            annotfileattachment.setMarkUpTitle("ポップアップウィンドウのタイトル文字列");

            // サブジェクトの短い説明設定
            annotfileattachment.setMarkUpSubj("サブジェクトの短い説明設定");

            // 注釈生成日時の設定
            try (PtlDate dateMarkup = new PtlDate(2000, 4, 1, 23, 59, 59))
            {
                annotfileattachment.setMarkUpDate(dateMarkup);
            }

            // 不透明度を設定 0.0 ～ 1.0。0.0が透明、1.0が不透明
            annotfileattachment.setMarkUpCA(1.0f);

            // 添付ファイル名設定
            annotfileattachment.setFileName(attachmentfilename);

            // 添付するファイルの読み込み
            try (PtlParamInput attachmentfile = new PtlParamInput(attachmentfilename)) {
                annotfileattachment.readFile(attachmentfile);
            }

            // ポップアップ
            // 矩形座標を設定 座標の単位はmmで原点(0,0)は左下
            try (PtlRect rectPopup = new PtlRect(100.0f, 100.0f, 150.0f, 150.0f))
            {
                annotpopup.setRect(rectPopup);
            }

            // オープン状態を設定 true=オープン状態
            annotpopup.setOpenState(true);

            // ポップアップ注釈を設定
            annotfileattachment.setAnnotPopUp(annotpopup);

            // 注釈の追加
            annots.append(annotfileattachment);
        }
    }
}
