/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：リンク注釈の作成
  　　　入力PDFの1ページ目に各種リンク注釈を挿入してPDF形式で出力する。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/


package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;

public class AppendAnnotLink {
    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.print("usage: java AppendAnnotLink in-pdf-file");
        System.out.println(" out-pdf-file アクションの種類");
        System.out.println("アクションの種類");
        System.out.println("0 : GOTOアクションの設定");
        System.out.println("1 : GOTORアクションの設定");
        System.out.println("2 : Launchアクションの設定");
        System.out.println("3 : URIアクションの設定");
    }

    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage();
            return;
        }


        // コマンドライン引数の取得
        int actionKind = Integer.parseInt(args[2]);
        if((actionKind < 0) || (3 < actionKind)) {
            System.out.print("アクションの種類 ");
            System.out.println("は0から3の範囲の整数値で指定してください。");
            printUsage();
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ページコンテナが空\n");
                    return;
                }

                try (PtlPage page = pages.get(0)) {//1ページ目の取得
                    // リンク注釈追加
                    addAnnotLink(br, pages, page, actionKind);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void addAnnotLink(BufferedReader br, PtlPages pages,
                                    PtlPage page, int actionKind)
        throws PtlException, Exception, Error {
        try (PtlAnnots annots = page.getAnnots(); //注釈コンテナの取得
             PtlAnnotLink annotlink = new PtlAnnotLink()) {//PDFのリンク注釈
            // 矩形座標を設定
            try (PtlRect rect = new PtlRect()) {
                annotlink.setRect(setRectCoordinate(br, rect));
            }
            // 内容を設定
            annotlink.setTextContents("挿入されるリンク注釈");
            // アクションの設定
            setAction(pages, annotlink, actionKind);
            // 注釈の追加
            annots.append(annotlink);
        }
    }

    public static void setAction(PtlPages pages, PtlAnnotLink annotlink, int actionKind)
        throws PtlException, Exception, Error {
        switch (actionKind) {
        case 0:
            // GOTOアクションの設定
            try (PtlActionGoTo acttiongoto = new PtlActionGoTo();
                 PtlDestFit destfit = new PtlDestFit()) {//宛先
                // 宛先ページの設定（最終ページに）
                destfit.setPageNumber(pages.getCount() - 1);
                // 宛先の設定
                acttiongoto.setDest(destfit);
                // アクションの設定
                annotlink.setAction(acttiongoto);
            }
            break;
        case 1:
            // GOTORアクションの設定
            try (PtlActionGoToR actiongotor = new PtlActionGoToR();
                 PtlDestFit destfit = new PtlDestFit()) {//宛先
                // 宛先ページの設定（最終ページに）
                destfit.setPageNumber(pages.getCount() - 1);
                // 宛先の設定
                actiongotor.setDest(destfit);
                // ファイル間移動用PDFファイルを設定
                actiongotor.setFileName("test.pdf");
                // 新ウィンドウフラグを設定
                actiongotor.setNewWindowFlag(true);
                // アクションの設定
                annotlink.setAction(actiongotor);
            }
            break;
        case 2:
            // Launchアクションの設定
            try (PtlActionLaunch actionlaunch = new PtlActionLaunch()) {
                // 起動ファイル名を設定
                actionlaunch.setFileName("test.txt");
                // 新ウィンドウフラグを設定
                actionlaunch.setNewWindowFlag(true);
                // アクションの設定
                annotlink.setAction(actionlaunch);
            }
            break;
        case 3:
            // URIアクションの設定
            try (PtlActionURI actionurl = new PtlActionURI()) {
                // URIを設定
                actionurl.setURI("https://www.antenna.co.jp/");
                // アクションの設定
                annotlink.setAction(actionurl);
            }
            break;
        }
    }

    /**
     * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
     * 原点はPDFの左下端。
     * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * @param br BufferedReader。数値の読み取りに使う。
     * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
     * @return 指定したleft, bottom, right, topの数値を持つPtlRect
     * @throws IOException
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    public static PtlRect setRectCoordinate(BufferedReader br, PtlRect outputRect)
           throws IOException, PtlException, Exception, Error {
        float top, bottom, left, right;
        boolean isValueOkay = false;
        
        while(!isValueOkay) {
            System.out.println("指定する矩形の各数値を入力してください。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) {//不正矩形は再入力させる
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            if(right < left) {//不正矩形は再入力させる
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setLeft(left);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setTop(top);
        }
        return outputRect;
    }
}
