/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：カスタムスタンプ注釈の作成

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class AppendAnnotStampCustom {
    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java AppendAnnotStampCustom in-pdf-file" +
                           " out-pdf-file スタンプ種類 custom-icon-file");
        System.out.println("スタンプ種類");
        System.out.println("0 : カスタム(PDF) 1 : カスタム(Image)");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            printUsage();
            return;
        }

        String stampKind = args[2];
        switch (stampKind) {
        case "0":
        case "1":
            break;
        default:
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlPages pages = doc.getPages()) { //ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ページコンテナが空\n");
                    return;
                }

                try (PtlPage page = pages.get(0);
                     PtlAnnots annots = page.getAnnots()) {
                    switch (stampKind) {
                    case "0":
                        // 読み込んだPDFを外観とするスタンプ
                        if (args.length < 4) {
                            System.out.println("スタンプ注釈に使うpdfを指定してください。");
                            printUsage();
                            return;
                        }
                        addCustomStampAnnotFromPdf(annots, args[3]);
                        break;
                    case "1":
                        // 読み込んだ画像を外観とするスタンプ
                        if (args.length < 4) {
                            System.out.println("スタンプ注釈に使う画像を指定してください。");
                            printUsage();
                            return;
                        }
                        addCustomStampAnnotFromImage(annots, args[3]);
                        break;
                    }
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void addCustomStampAnnotFromPdf(PtlAnnots annots, String pathPdf)
        throws PtlException, Exception, Error {
        try (PtlAnnotStamp stampAnnot = new PtlAnnotStamp();
             PtlPDFDocument doc_custom = new PtlPDFDocument();
             PtlParamInput inputCustom = new PtlParamInput(pathPdf)) {    // PDFファイル
            stampAnnot.setIconType(PtlAnnotStamp.ICON_TYPE.ICON_CUSTOM);
            stampAnnot.setAnnotFlags(PtlAnnot.FLAG_PRINT);

            // スタンプにするPDFファイルをロードします。
            doc_custom.load(inputCustom);

            try (PtlPages pagesCustomStamp = doc_custom.getPages()) {
                // ページコンテナが空かどうか
                if (pagesCustomStamp.isEmpty()) {
                    System.out.println("ページコンテナが空\n");
                    return;
                }

                // 追加された画像ページを取得する
                try (PtlPage pageCustomStamp = pagesCustomStamp.get(0)) {
                    // 画像ページを注釈に追加する
                    stampAnnot.setPage(pageCustomStamp);

                    try (PtlSize size = pageCustomStamp.getSize();
                         PtlRect rect = new PtlRect(10.0f, 270.0f,
                                                    10.0f+size.getWidth(),
                                                    270.0f+size.getHeight())) {
                        stampAnnot.setRect(rect);
                    }
                }
            }

            stampAnnot.setIconName("MyIcon");

            annots.append(stampAnnot);
        }
    }

    public static void addCustomStampAnnotFromImage(PtlAnnots annots, String pathImage)
        throws PtlException, Exception, Error {
        try (PtlAnnotStamp stampAnnot = new PtlAnnotStamp()) {

            stampAnnot.setIconType(PtlAnnotStamp.ICON_TYPE.ICON_CUSTOM);
            stampAnnot.setAnnotFlags(PtlAnnot.FLAG_PRINT);
            // 画像描画パラメータ
            // 画像ファイル
            // 画像ページパラメータ の順に取得する
            try (PtlParamDrawImage paramDrawImage = new PtlParamDrawImage();
                 PtlParamInput inputCustom = new PtlParamInput(pathImage);     
                 PtlParamImagePage paramImagePage = new PtlParamImagePage()) {
                // 画像描画パラメータに画像ファイルを設定
                paramDrawImage.setImageStream(inputCustom);

                // 画像ページパラメータに画像描画パラメータを設定
                paramImagePage.setImage(paramDrawImage);

                // 画像ページのサイズを画像サイズにあわせる
                paramImagePage.setPaperType(PtlParamImagePage.PAPER_TYPE.PAPER_IMAGE_SIZE);

                // 画像ページパラメータから作成したページ
                try (PtlPage pageCustomStamp = new PtlPage(paramImagePage)) {
                    // 画像ページを注釈に追加する
                    stampAnnot.setPage(pageCustomStamp);

                    try (PtlSize size = pageCustomStamp.getSize();
                         PtlRect rect = new PtlRect(10.0f, 270.0f,
                                                    10.0f+size.getWidth(),
                                                    270.0f+size.getHeight())) {
                        stampAnnot.setRect(rect);
                    }
                }
            }

            stampAnnot.setIconName("MyIcon");

            annots.append(stampAnnot);
        }
    }

}
