/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：画像ファイルのPDF化(配置矩形の設定)

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;

public class ImageToPdfSetRect {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java ImageToPdfSetRect image-file" +
                           " out-pdf-file image-align");
        System.out.println("image-align : 画像の配置");
     }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 2) {
            printUsage();
            return;
        }

        //コマンドライン引数の読み取り
        String AlignType = args[2];
        try (PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();  //PDF文書を表現したクラスを新規作成
             PtlPages pages = doc.getPages()) {  //作成したクラスのページコンテナの取得

            // 画像描画パラメータ
            // 画像ファイル
            // 画像ページパラメータ の取得
            try (PtlParamDrawImage paramDrawImage = new PtlParamDrawImage();
                 PtlParamInput inputImage = new PtlParamInput(args[0]);
                 PtlParamImagePage paramImagePage = new PtlParamImagePage();
                 PtlRect rect = new PtlRect();
                 BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
                // 画像の配置を設定
                try{
                    paramImagePage.setAlign(
                        PtlParamImagePage.ALIGN.valueOf(AlignType));
                }catch (IllegalArgumentException iaex){
                    //PtlParamImagePage.PAPER_TYPEの指定が誤っていた場合のエラーメッセージ
                    System.out.println(iaex.getMessage());
                    System.out.println("ERROR : [画像の配置]には" +
                                       "PtlParamImagePage.ALIGNに" +
                                       "含まれる名前を指定してください。");
                    iaex.printStackTrace();
                }
                //画像の配置矩形を設定
                paramImagePage.setRect(setRectCoordinate(br, rect));

                // 画像描画パラメータに画像ファイルを設定
                paramDrawImage.setImageStream(inputImage);
                // 画像ページパラメータに画像描画パラメータを設定
                paramImagePage.setImage(paramDrawImage);
                // 画像ページのサイズを画像サイズにあわせる
                paramImagePage.setPaperType(PtlParamImagePage.PAPER_TYPE.PAPER_A4);
                // ページコンテナに画像ページパラメータを追加
                pages.append(paramImagePage);
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
             System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                                "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    /**
     * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
     * 原点はPDFの左下端。
     * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * @param br BufferedReader。数値の読み取りに使う。
     * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
     * @return 指定したleft, bottom, right, topの数値を持つPtlRect
     * @throws IOException
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    public static PtlRect setRectCoordinate(BufferedReader br, PtlRect outputRect)
           throws IOException, PtlException, Exception, Error {
        float top, bottom, left, right;
        boolean isValueOkay = false;
        
        while(!isValueOkay) {
            System.out.println("指定する矩形の各数値を入力してください。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) {//不正矩形は再入力させる
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            if(right < left) {//不正矩形は再入力させる
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setLeft(left);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setTop(top);
        }
        return outputRect;
    }
}
