/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：画像ファイルのPDF化と既存PDFへのページ追加

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class InsertImagePage {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java InsertImagePage in-pdf-file" +
                           " out-pdf-file insert-image-file page-to-insert" +
                           " [multi-tiff-page]");
        System.out.println("page-to-insert : ");
        System.out.println("0 : 先頭ページに挿入　それ以外：そのページ番号の後ろに挿入");
        System.out.println("[multi-tiff-page] : マルチTiffページ番号の設定" +
                           " (マルチページTiff挿入時未設定なら全ページを挿入)");
        System.out.println("-1 : 全ページを挿入　それ以外：ページ番号");
    }


    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            printUsage();
            return;
        }

        // コマンドライン引数の読み取り（マルチページTIFFのページ指定）
        int pageNumOfTiff = -1;
        if (args.length == 5) {
            pageNumOfTiff = Integer.parseInt(args[4]);
        }
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            // 挿入ページ指定の判定
            int pageToInsert = Integer.parseInt(args[3]);
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);
            if((numPages < 0)||(numPages < pageToInsert))
            {
                System.err.println("挿入したいページは全ページ数以下で" +
                                   "0より大きい値を指定してください。");
                return;
            }
            // 画像描画パラメータ
            // 画像ファイル
            // 画像ページパラメータ の取得
            try (PtlPages pages = doc.getPages()) { // ページコンテナの取得

                
                try (PtlParamDrawImage paramDrawImage = new PtlParamDrawImage();
                     PtlParamInput inputImage = new PtlParamInput(args[2]);
                     PtlParamImagePage paramImagePage = new PtlParamImagePage()) {
                    // 画像描画パラメータに画像ファイルを設定
                    paramDrawImage.setImageStream(inputImage);

                    // マルチTiffのページ番号の設定（Tiffファイルにのみ有効） 
                    // -1を設定すると全ページを挿入します。
                    // 設定しない場合は、-1が設定されたとみなします
                    paramDrawImage.setImagePageNumber(pageNumOfTiff);

                    // 画像ページパラメータに画像描画パラメータを設定
                    paramImagePage.setImage(paramDrawImage);

                    // 画像ページのサイズをA4にする
                    paramImagePage.setPaperType(PtlParamImagePage.PAPER_TYPE.PAPER_A4);

                    // 画像を配置するときの余白の設定
                    paramImagePage.setMargin(20.0f, 15.0f, 20.0f, 15.0f);

                    // 画像の配置の設定 ALIGN_TOP_LEFT = 1, // 左上 
                    paramImagePage.setAlign(PtlParamImagePage.ALIGN.ALIGN_TOP_LEFT);

                    // ページコンテナに画像ページパラメータを追加
                    // insert()のパラメータindexは0が先頭のため1を引く
                    pages.insert(pageToInsert , paramImagePage);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
}
