/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：開き方の設定(ページ関連の設定)

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class SetOpenModeOfPage {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.print("usage: java SetOpenModeOfPage in-pdf-file out-pdf-file");
        System.out.println(" ページモード種別 ページレイアウト種別");
        System.out.println("ページモード種別: ");
        System.out.println("0 : ページのみ表示 1 : アウトラインパネルとページ");
        System.out.println("2 : サムネイルパネルとページ 3 : レイヤーパネルとページ");
        System.out.println("4 : 添付ファイルパネルとページ");
        System.out.println("ページレイアウト種別: ");
        System.out.println("0 : 単一ページ");
        System.out.println("1 : 連続ページ");
        System.out.println("2 : 奇数ページを左側にして連続見開きページ");
        System.out.println("3 : 奇数ページを右側にして連続見開きページ");
        System.out.println("4 : 奇数ページを左側にして見開きページ");
        System.out.println("5 : 奇数ページを右側にして見開きページ");
    }


    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            printUsage();
            return;
        }
        // コマンドライン引数の読み取り
        // ページモードの選択
        PtlOpenMode.PAGE_MODE pageModeType = PtlOpenMode.PAGE_MODE.PAGE_MODE_USE_NONE;
        switch(args[2]) {
        case "0": //0 : ページのみ表示
            pageModeType = PtlOpenMode.PAGE_MODE.PAGE_MODE_USE_NONE;
            break;
        case "1": //1 : アウトラインパネルとページ
            pageModeType = PtlOpenMode.PAGE_MODE.PAGE_MODE_USE_OUTLINES;
            break;
        case "2": //2 : サムネイルパネルとページ
            pageModeType = PtlOpenMode.PAGE_MODE.PAGE_MODE_USE_THUMBS;
            break;
        case "3": //3 : レイヤーパネルとページ
            pageModeType = PtlOpenMode.PAGE_MODE.PAGE_MODE_USE_OC;
            break;
        case "4": //4 : 添付ファイルパネルとページ
            pageModeType = PtlOpenMode.PAGE_MODE.PAGE_MODE_USE_ATTACHMENTS;
            break;
        default :
            System.out.println("ページモード種別は0から4の整数で指定して下さい。");
            printUsage();
            break;
        }

        // ページレイアウト種別:
        PtlOpenMode.PAGE_LAYOUT pageLayoutType = PtlOpenMode.PAGE_LAYOUT.PAGE_LAYOUT_SINGLE_PAGE;
        switch(args[3]) {
        case "0": //0 : 単一ページ
            pageLayoutType = PtlOpenMode.PAGE_LAYOUT.PAGE_LAYOUT_SINGLE_PAGE;
            break;
        case "1": //1 : 連続ページ
            pageLayoutType = PtlOpenMode.PAGE_LAYOUT.PAGE_LAYOUT_ONE_COLUMN;
            break;
        case "2": //2 : 奇数ページを左側にして連続見開きページ
            pageLayoutType = PtlOpenMode.PAGE_LAYOUT.PAGE_LAYOUT_TWO_COLUMN_LEFT;
            break;
        case "3": //3 : 奇数ページを右側にして連続見開きページ
            pageLayoutType = PtlOpenMode.PAGE_LAYOUT.PAGE_LAYOUT_TWO_COLUMN_RIGHT;
            break;
        case "4": //4 : 奇数ページを左側にして見開きページ
            pageLayoutType = PtlOpenMode.PAGE_LAYOUT.PAGE_LAYOUT_TWO_PAGE_LEFT;
            break;
        case "5": //5 : 奇数ページを右側にして見開きページ
            pageLayoutType = PtlOpenMode.PAGE_LAYOUT.PAGE_LAYOUT_TWO_PAGE_RIGHT;
            break;
        default :
            System.out.println("ページレイアウト種別は0から5の整数で指定して下さい。");
            printUsage();
            break;
        }


        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlDocProperty docProperty = doc.getDocProperty();  // PDFの文書プロパティ
                 PtlOpenMode openmode = docProperty.getOpenMode()) { // 開き方の取得
                /*----- 開き方のデフォルト設定 -----*/
                // オープンアクションの設定
                try (PtlActionGoTo act = new PtlActionGoTo(); // 設定するGoToアクション
                     PtlDestFitR dst = new PtlDestFitR()) {   // PDFの宛先
                    // 宛先ページの設定
                    // 最初に開くページを2ページ目にセット(ページ番号は0ベース)
                    dst.setPageNumber(1);
                    // 開く座標の設定
                    // leftの設定
                    dst.setLeft(100.0f);
                    //dst.setLeftNull();
                    // bottomの設定
                    dst.setBottom(100.0f);
                    //dst.setBottomNull();
                    // rightの設定
                    dst.setRight(120.0f);
                    //dst.setRightNull();
                    // topの設定
                    dst.setTop(150.0f);
                    //dst.setTopNull();
                    // zoomの設定
                    dst.setZoom(100.0f);
                    //dst.setZoomNull();

                    act.setDest(dst);
                    openmode.setOpenAction(act);
                }

                // ユーザーインターフェイスオプションを設定（論理和）
                // UI_OPTION_HIDE_WINDOWUI = 0x00000004
                /* ウインドウコントロールを非表示 */
                openmode.setUIOption(PtlOpenMode.UI_OPTION_HIDE_WINDOWUI);

                // ウインドウオプションを設定（論理和）
                // WINDOWS_OPTION_FIT_WINDOW = 0x00000001,
                /* ページにウィンドウサイズを合あわせる */
                // WINDOWS_OPTION_CENTER_WINDOW = 0x00000002,
                /* ウィンドウを画面中央に配置 */
                openmode.setWindowOption(PtlOpenMode.WINDOWS_OPTION_FIT_WINDOW
                    |PtlOpenMode.WINDOWS_OPTION_CENTER_WINDOW);

                // ウィンドウのタイトルバーに文書タイトルを表示するかどうか設定
                //（true: 非表示、false: 表示）
                openmode.setDisplayDocTitle(true);

                // 読み上げオプション設定 DIRECTION_R2L = 2
                /* 右から左(中国語、日本語、韓国語のような縦書きも含む)    */
                docProperty.setDirection(PtlDocProperty.DIRECTION.DIRECTION_R2L);
                /*----- 以上、ここまでが開き方のデフォルト設定 -----*/

                
                // ページモードの設定 ページレイアウトの設定
                // (デフォルト設定よりも優先させるために後で設定する)
                openmode.setPageMode(pageModeType);
                openmode.setPageLayout(pageLayoutType);

            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
    
}
