/*
  Antenna House PDF Tool API V5.0
  Java Interface sample program

  概要：開き方の設定(UI表示関連の設定)

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class SetOpenModeOfWindow {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.print("usage: java SetOpenModeOfPageOfWindow in-pdf-file out-pdf-file");
        System.out.println(" fit-window center-window full-screen");
        System.out.println("fit-window : ページにウィンドウサイズを合わせる" +
                           " [0 でフラグoff, 1 でフラグon]");
        System.out.println("center-window : ウィンドウを画面中央に配置" +
                           " [0 でフラグoff, 1 でフラグon]");
        System.out.println("full-screen : フルスクリーンモードで開く" +
                           " [0 でフラグoff, 1 でフラグon]");
    }


    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 5) {
            printUsage();
            return;
        }
        // コマンドライン引数の読み取り
        boolean flagFitWindow;
        boolean flagCenterWindow;
        boolean flagFullScreen;
        int windowOption;
        // fit-window 
        try {
            flagFitWindow = readBoolArgs(args[2], "fit-windowは" +
                                           "0か1で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }
        // center-window
        try {
            flagCenterWindow = readBoolArgs(args[3], "center-windowは" +
                                           "0か1で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }
        // full-screen
        try {
            flagFullScreen = readBoolArgs(args[4], "full-screenは" +
                                         "0か1で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }
        
        // 各フラグの論理和をWindowOptionに設定。
        windowOption = PtlOpenMode.WINDOWS_OPTION_NONE;
        if(flagFitWindow) {
            windowOption = windowOption | PtlOpenMode.WINDOWS_OPTION_FIT_WINDOW;
        }
        if(flagCenterWindow) {
            windowOption = windowOption | PtlOpenMode.WINDOWS_OPTION_CENTER_WINDOW;
        }
        if(flagFullScreen) {
            windowOption = windowOption | PtlOpenMode.WINDOWS_OPTION_FULL_SCREEN;
        }


        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlDocProperty docProperty = doc.getDocProperty();  // PDFの文書プロパティ
                 PtlOpenMode openmode = docProperty.getOpenMode()) { // 開き方の取得
                /*----- 開き方のデフォルト設定 -----*/
                // ページモードの設定 PAGE_MODE_USE_THUMBS = 3  サムネイルパネルとページ
                openmode.setPageMode(PtlOpenMode.PAGE_MODE.PAGE_MODE_USE_THUMBS);

                // ページレイアウトの設定 PAGE_LAYOUT_ONE_COLUMN = 2 /* 連続ページ */
                openmode.setPageLayout(PtlOpenMode.PAGE_LAYOUT.PAGE_LAYOUT_ONE_COLUMN);

                // オープンアクションの設定
                try (PtlActionGoTo act = new PtlActionGoTo(); // 設定するGoToアクション
                     PtlDestFitR dst = new PtlDestFitR()) {   // PDFの宛先
                    // 宛先ページの設定
                    // 最初に開くページを2ページ目にセット(ページ番号は0ベース)
                    dst.setPageNumber(1);
                    // 開く座標の設定
                    // leftの設定
                    dst.setLeft(100.0f);
                    //dst.setLeftNull();
                    // bottomの設定
                    dst.setBottom(100.0f);
                    //dst.setBottomNull();
                    // rightの設定
                    dst.setRight(120.0f);
                    //dst.setRightNull();
                    // topの設定
                    dst.setTop(150.0f);
                    //dst.setTopNull();
                    // zoomの設定
                    dst.setZoom(100.0f);
                    //dst.setZoomNull();

                    act.setDest(dst);
                    openmode.setOpenAction(act);
                }

                // ユーザーインターフェイスオプションを設定（論理和）
                // UI_OPTION_HIDE_WINDOWUI = 0x00000004
                /* ウインドウコントロールを非表示 */
                openmode.setUIOption(PtlOpenMode.UI_OPTION_HIDE_WINDOWUI);


                // ウィンドウのタイトルバーに文書タイトルを表示するかどうか設定
                //（true: 非表示、false: 表示）
                openmode.setDisplayDocTitle(true);

                // 読み上げオプション設定 DIRECTION_R2L = 2
                /* 右から左(中国語、日本語、韓国語のような縦書きも含む)    */
                docProperty.setDirection(PtlDocProperty.DIRECTION.DIRECTION_R2L);
                /*----- 以上、ここまでが開き方のデフォルト設定 -----*/

                // ウインドウオプションを設定（論理和）
                // (デフォルト設定よりも優先させるために後で設定する)
                openmode.setWindowOption(windowOption);
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }


    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
        throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }    
}
