/*
  Antenna House PDF Tool API V6.0
  Java Interface sample program

  概要：注釈情報を取得してマークアップ注釈か否かを判別して情報を表示する。
  　　　ポップアップ注釈が含まれていた場合はそれの情報を表示する。

  Copyright (C) 2018- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class ShowAnnotMarkupInfo {
    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java ShowAnnotMarkupInfo in-pdf-file");
    }



    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 1) {
            printUsage();
            return;
        }
        String outpathAttach = "";
        if (args.length == 2) {
            outpathAttach = args[1];
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {

            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlPages pages = doc.getPages()) { //ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ページコンテナが空\n");
                    return;
                }

                int numPages = pages.getCount();
                for(int i=0; i<numPages; i++) {
                    System.out.println("ページ" + (i+1));

                    try (PtlPage page = pages.get(i)) {
                        // 注釈コンテナが空かどうか
                        if (!page.hasAnnots()) {
                            System.out.println("注釈なし");
                            continue;
                        }

                        try (PtlAnnots annots = page.getAnnots()) {
                            // 注釈数の取得
                            int numAnnots = annots.getCount();
                            System.out.println("注釈数 : " + numAnnots);
                            for(int j=0; j<numAnnots; j++) {
                                System.out.println("index num : " + j +
                                                   "について、マークアップ注釈か否かを" +
                                                   "判定します。");
                                try (PtlAnnot annot = annots.get(j)) {
                                    if (annot.isMarkup()) {
                                        System.out.println("マークアップ注釈です。" +
                                                           "情報を表示します。");
                                        showAnnotMarkup(annot);
                                    }else {
                                        System.out.println("マークアップ注釈では" +
                                                           "ありません。");
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }


    public static void showAnnotMarkup(PtlAnnot annot)
        throws PtlException, Exception, Error {
        PtlAnnotMarkup annotMarkup = (PtlAnnotMarkup)annot;
        System.out.println("MarkUpTitle = \"" + annotMarkup.getMarkUpTitle() + "\"" );
        System.out.println("MarkUpSubj = \"" + annotMarkup.getMarkUpSubj() + "\"" );
        try (PtlDate date = annotMarkup.getMarkUpDate())
        {
            int year = date.getYear();
            int month = date.getMonth();
            int day = date.getDay();
            int hour = date.getHour();
            int min = date.getMin();
            int sec = date.getSec();
            System.out.printf("MarkUpDate : %d/%d/%d %d:%d:%d\n", year, month, day, hour,
                              min, sec);
        }
        System.out.printf("MarkUpCA : %f\n", annotMarkup.getMarkUpCA());
        if (annotMarkup.hasAnnotPopup()) {
            try (PtlAnnotPopup popup = annotMarkup.getAnnotPopup())
            {
                System.out.println("ポップアップ注釈が含まれていました。" +
                                   "その情報を表示します。");
                showPopup(popup);
            }
        }
    }

    public static void showPopup(PtlAnnot annot)
        throws PtlException, Exception, Error {
        System.out.println("ANNOT_TYPE = Popup");
        PtlAnnotPopup annotPopup = (PtlAnnotPopup)annot;
        showAnnotCommon(annotPopup);
        boolean openState = annotPopup.getOpenState();
        if (openState)
            System.out.println("OPEN_STATE == true" );
        else
            System.out.println("OPEN_STATE == false" );
    }

    public static void showAnnotCommon(PtlAnnot annot)
        throws PtlException, Exception, Error {
        try (PtlRect rect = annot.getRect()) {
            System.out.printf("Rcet = %f,%f,%f,%f\n" , rect.getLeft(), rect.getBottom(),
                              rect.getRight(), rect.getTop());
        }
        System.out.println("TextContents = \"" + annot.getTextContents() + "\"" );
        try (PtlDate date = annot.getDate())
        {
            int year = date.getYear();
            int month = date.getMonth();
            int day = date.getDay();
            int hour = date.getHour();
            int min = date.getMin();
            int sec = date.getSec();
            System.out.printf("Date : %d/%d/%d %d:%d:%d\n", year, month, day, hour,
                              min, sec);
        }
        int flags = annot.getAnnotFlags();
        if ((flags & PtlAnnot.FLAG_INVISIBLE) == PtlAnnot.FLAG_INVISIBLE)
            System.out.println("FLAG_INVISIBLE on" );
        else
            System.out.println("FLAG_INVISIBLE off" );
        if ((flags & PtlAnnot.FLAG_HIDDEN) == PtlAnnot.FLAG_HIDDEN)
            System.out.println("FLAG_HIDDEN on" );
        else
            System.out.println("FLAG_HIDDEN off" );
        if ((flags & PtlAnnot.FLAG_PRINT) == PtlAnnot.FLAG_PRINT)
            System.out.println("FLAG_PRINT on" );
        else
            System.out.println("FLAG_PRINT off" );
        if ((flags & PtlAnnot.FLAG_NOZOOM) == PtlAnnot.FLAG_NOZOOM)
            System.out.println("FLAG_NOZOOM on" );
        else
            System.out.println("FLAG_NOZOOM off" );
        if ((flags & PtlAnnot.FLAG_NOROTATE) == PtlAnnot.FLAG_NOROTATE)
            System.out.println("FLAG_NOROTATE on" );
        else
            System.out.println("FLAG_NOROTATE off" );
        if ((flags & PtlAnnot.FLAG_NOVIEW) == PtlAnnot.FLAG_NOVIEW)
            System.out.println("FLAG_NOVIEW on" );
        else
            System.out.println("FLAG_NOVIEW off" );
        if ((flags & PtlAnnot.FLAG_READONLY) == PtlAnnot.FLAG_READONLY)
            System.out.println("FLAG_READONLY on" );
        else
            System.out.println("FLAG_READONLY off" );
        if ((flags & PtlAnnot.FLAG_LOCKED) == PtlAnnot.FLAG_LOCKED)
            System.out.println("FLAG_LOCKED on" );
        else
            System.out.println("FLAG_LOCKED off" );
        if ((flags & PtlAnnot.FLAG_TOGGLE_NOVIEW) == PtlAnnot.FLAG_TOGGLE_NOVIEW)
            System.out.println("FLAG_TOGGLE_NOVIEW on" );
        else
            System.out.println("FLAG_TOGGLE_NOVIEW off" );
        if ((flags & PtlAnnot.FLAG_LOCKED_CONTENTS) == PtlAnnot.FLAG_LOCKED_CONTENTS)
            System.out.println("FLAG_LOCKED_CONTENTS on" );
        else
            System.out.println("FLAG_LOCKED_CONTENTS off" );
        try (PtlColor color = annot.getColor()) {
            System.out.println("show color info:");
            showColor(color);
        }
        try (PtlColor colorI = annot.getInteriorColor()) {
            System.out.println("show interior color info:");
            showColor(colorI);
        }
        PtlAnnot.BORDER_STYLE boderStyle = annot.getBorderStyle();
        switch (boderStyle) {
        case BORDER_SOLID:
            System.out.println("BORDER_STYLE = SOLID" );
            break;
        case BORDER_DASHED:
            System.out.println("BORDER_STYLE = DASHED" );
            break;
        case BORDER_BEVELED:
            System.out.println("BORDER_STYLE = BEVELED" );
            break;
        case BORDER_INSET:
            System.out.println("BORDER_STYLE = INSET" );
            break;
        case BORDER_UNDERLINE:
            System.out.println("BORDER_STYLE = UNDERLINE" );
            break;
        }
        PtlAnnot.BORDER_LINE_WIDTH boderWidth = annot.getBorderWidth();
        switch (boderWidth) {
        case BORDER_WIDTH_NONE:
            System.out.println("BORDER_WIDTH = NONE" );
            break;
        case BORDER_WIDTH_THIN:
            System.out.println("BORDER_WIDTH = THIN" );
            break;
        case BORDER_WIDTH_MIDDLE:
            System.out.println("BORDER_WIDTH = MIDDLE" );
            break;
        case BORDER_WIDTH_THICK:
            System.out.println("BORDER_WIDTH = THICK" );
            break;
        }
    }
    
    public static void showColor(PtlColor color)
        throws PtlException, Exception, Error {
        PtlColor.COLOR_TYPE colorType = color.getType();
        switch (colorType) {
        case TYPE_NONE:
            System.out.println("COLOR_TYPE = None" );
            break;
        case TYPE_DEVICE_GRAY:
            PtlColorDeviceGray colorGray = (PtlColorDeviceGray)color;
            System.out.printf("COLOR_TYPE = GRAY(%f)\n" , colorGray.getGray());
            break;
        case TYPE_DEVICE_RGB:
            PtlColorDeviceRGB colorRGB = (PtlColorDeviceRGB)color;
            System.out.printf("COLOR_TYPE = RGB(%f,%f,%f)\n" , colorRGB.getR(),
                              colorRGB.getG(), colorRGB.getB());
            break;
        case TYPE_DEVICE_CMYK:
            PtlColorDeviceCMYK colorCMYK = (PtlColorDeviceCMYK)color;
            System.out.printf("COLOR_TYPE = CMYK(%f,%f,%f,%f)\n" , colorCMYK.getC(),
                              colorCMYK.getM(), colorCMYK.getY(), colorCMYK.getK());
            break;
        }
    }

    
}
