﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：指定した画像の角度を変えて描画する

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

   Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace AddImageSetAngle
{
    internal class AddImageSetAngle
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: AddImageSetAngle in-pdf-file out-pdf-file insert-image-file angle");
            Console.WriteLine("angle : 画像の角度");
        }

        static void Main(string[] args)
        {
            if (args.Length < 3)
            {
                printUsage();
                return;
            }
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    //コマンドライン引数の取得
                    float angle = float.Parse(args[3]);

                    // ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空");
                            return;
                        }

                        try
                        {
                            using (PtlPage page = pages.get(0))     // 先頭ページの取得(パラメータindexは0が先頭)
                            using (PtlContent content = page.getContent())      // ページコンテントの取得
                            using (PtlRect outputRect = page.getViewBox())      // 出力矩形の設定。(ページの大きさを得るためにViewBoxを取得する)
                            using (PtlParamDrawImage paramDrawImage = new PtlParamDrawImage())  //画像の描画に使うパラメータクラス
                            using (PtlParamInput insertImage = new PtlParamInput(args[2]))      //画像指定に使うパラメータクラス
                            using (PtlParamWriteString plainParam = new PtlParamWriteString())  // 文字描画のパラメータクラス。
                            {
                                //入力画像の角度を設定
                                paramDrawImage.setAngle(angle);

                                //入力画像ストリームの設定
                                paramDrawImage.setImageStream(insertImage);
                                //マルチTiffのページ番号の設定（マルチTiffファイルにのみ有効） 
                                //数値を設定しない場合は、-1が設定されたとみなします
                                //PtlParamImagePageに設定する場合に限り、-1を設定すると全ページを挿入します。
                                paramDrawImage.setImagePageNumber(0);

                                // 透明度を設定
                                paramDrawImage.setOpacity(1);
                                // 画像出力
                                content.drawImage(outputRect, PtlContent.ALIGN.ALIGN_CENTER, paramDrawImage);
                                // angleの状態テキストを描画
                                content.writeString(outputRect, PtlContent.ALIGN.ALIGN_CENTER, "angle = " + angle, plainParam);
                            }
                        }
                        catch (ArgumentException ex)
                        {
                            //PtlContent.ALIGN指定のエラー処理
                            Console.WriteLine(ex.Message);
                            Console.WriteLine("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                            Console.WriteLine(ex.StackTrace);
                        }
                    }
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (FileNotFoundException fnfx)
            {
                Console.WriteLine("FileNotFoundException : ErrorCode = " + fnfx.Message + "\n  " + fnfx.StackTrace);
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }
    }
}
