﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：指定したページの指定した位置にPDFの1ページを挿入します。
  　　　挿入する際に指定した角度に傾けます。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace AddPdfOnPageSetAngle
{
    internal class AddPdfOnPageSetAngle
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: AddForm in-pdf-file out-pdf-file");
            Console.WriteLine("page-num insert-pdf-file insert-page-num align angle");
        }

        static void Main(string[] args)
        {
            if (args.Length < 1)
            {
                printUsage();
                return;
            }
            try
            {
                Console.WriteLine(args[0]);

                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    //コマンドライン引数の取得
                    int pageToAdd = int.Parse(args[2]);
                    String insertPdfURI = args[3];
                    int insertPageNum = int.Parse(args[4]);
                    String align = args[5];
                    float angle = float.Parse(args[6]);
                    // pageToAddの判定
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);
                    if ((numPages < 0) || (numPages < pageToAdd))
                    {
                        TextWriter errorWriter = Console.Error;
                        errorWriter.WriteLine("page-numは入力PDFの全ページ数よりも小さい正の値を" +
                                "指定してください。");
                        printUsage();
                        return;
                    }

                    // ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                      // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空");
                            return;
                        }
                        // 書き込み先ページを取得(パラメータindexは0が先頭のため1を引く)
                        using (PtlPage page = pages.get(pageToAdd - 1))
                        {
                            //フォーム（PDFのページ）の描画
                            drawFormSetAngle(page, insertPdfURI, insertPageNum, align, angle);
                        }
                    }
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (FileNotFoundException fnfx)
            {
                Console.WriteLine("FileNotFoundException : ErrorCode = " + fnfx.Message + "\n  " + fnfx.StackTrace);
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void drawFormSetAngle(PtlPage page, String insertPdfURI,
                                            int insertPageNum, String align, float angle)
        {
            try
            {
                using (PtlContent content = page.getContent())      // 挿入先ページコンテントの取得
                using (PtlParamInput insertPdf = new PtlParamInput(insertPdfURI))   // 挿入PDF指定に使うパラメータクラス
                using (PtlPDFDocument docToInsert = new PtlPDFDocument())   // 挿入PDFの実体
                using (PtlRect outputRect = new PtlRect())          // 出力矩形の初期化
                {
                    docToInsert.load(insertPdf);
                    int numPages = docToInsert.getPageCount();
                    if ((numPages < 0) || (numPages < insertPageNum))
                    {
                        TextWriter errorWriter = Console.Error;
                        errorWriter.WriteLine("insert-page-numは挿入するPDFの全ページ数よりも小さい正の値を指定してください。");
                        Console.WriteLine("挿入PDFページ数：" + numPages);
                        return;
                    }

                    using (PtlPages pagesToInsert = docToInsert.getPages())
                    using (PtlPage pageToInsert = pagesToInsert.get(insertPageNum - 1))
                    using (PtlParamDrawForm formInsert = new PtlParamDrawForm())
                    {
                        try
                        {
                            // ページの取得(パラメータindexは0が先頭のため1を引く)
                            formInsert.setPage(pageToInsert);
                            formInsert.setAngle(angle);
                            // 指定ページの挿入
                            content.drawForm(setRectCoordinate(outputRect), (PtlContent.ALIGN)valueAlign(align), formInsert);
                        }
                        catch (ArgumentException ex)
                        {//PtlContent.ALIGN指定のエラー処理
                            Console.WriteLine(ex.Message);
                            Console.WriteLine("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                            Console.WriteLine(ex.StackTrace);
                        }
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         * 特に外部からの呼び出しを想定しないためprivateとする。
         * @param br BufferedReader。数値の読み取りに使う。
         * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         * @throws IOException
         * @throws PtlException
         * @throws Exception
         */
        public static PtlRect setRectCoordinate(PtlRect outputRect)
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            try
            {
                while (!isValueOkay)
                {
                    Console.WriteLine("指定する矩形の各数値を入力してください。");
                    Console.Write("top (mm) : ");
                    top = float.Parse(Console.ReadLine());
                    Console.Write("bottom (mm) : ");
                    bottom = float.Parse(Console.ReadLine());
                    if (top < bottom)
                    {   //不正矩形は再入力させる
                        Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                        continue;
                    }
                    Console.Write("left (mm) : ");
                    left = float.Parse(Console.ReadLine());
                    Console.Write("right (mm) : ");
                    right = float.Parse(Console.ReadLine());
                    if (right < left)
                    {   //不正矩形は再入力させる
                        Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                        continue;
                    }

                    //矩形を正しく指定できた場合の処理
                    isValueOkay = true;
                    outputRect.setLeft(left);
                    outputRect.setBottom(bottom);
                    outputRect.setRight(right);
                    outputRect.setTop(top);
                }
            }
            catch (IOException ex)
            {
                throw ex;
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
            return outputRect;
        }

        // 文字列を PtlContent.ALIGN の数値型として返す
        static int valueAlign(String align)
        {
            if (align.Equals(PtlContent.ALIGN.ALIGN_TOP_LEFT.ToString()))
            {
                return (int)PtlContent.ALIGN.ALIGN_TOP_LEFT;
            }
            else if (align.Equals(PtlContent.ALIGN.ALIGN_TOP.ToString()))
            {
                return (int)PtlContent.ALIGN.ALIGN_TOP;
            }
            else if (align.Equals(PtlContent.ALIGN.ALIGN_TOP_RIGHT.ToString()))
            {
                return (int)PtlContent.ALIGN.ALIGN_TOP_RIGHT;
            }
            else if (align.Equals(PtlContent.ALIGN.ALIGN_LEFT.ToString()))
            {
                return (int)PtlContent.ALIGN.ALIGN_LEFT;
            }
            else if (align.Equals(PtlContent.ALIGN.ALIGN_CENTER.ToString()))
            {
                return (int)PtlContent.ALIGN.ALIGN_CENTER;
            }
            else if (align.Equals(PtlContent.ALIGN.ALIGN_RIGHT.ToString()))
            {
                return (int)PtlContent.ALIGN.ALIGN_RIGHT;
            }
            else if (align.Equals(PtlContent.ALIGN.ALIGN_BOTTOM_LEFT.ToString()))
            {
                return (int)PtlContent.ALIGN.ALIGN_BOTTOM_LEFT;
            }
            else if (align.Equals(PtlContent.ALIGN.ALIGN_BOTTOM.ToString()))
            {
                return (int)PtlContent.ALIGN.ALIGN_BOTTOM;
            }
            else if (align.Equals(PtlContent.ALIGN.ALIGN_BOTTOM_RIGHT.ToString()))
            {
                return (int)PtlContent.ALIGN.ALIGN_BOTTOM_RIGHT;
            }
            else
            {
                throw new ArgumentException();
            }
        }
    }
}
