﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：文字列の色を設定して１ページ目中央に描画するプログラム。
        このプログラムではセパレーション色[Separation]を用いて色を指定する。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace AddTextSetSeparationColor
{
    internal class AddTextSetSeparationColor
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: AddTextSetSeparationColor in-pdf-file out-pdf-file"
                             + " color-name value-c value-m value-y value-k tint");
        }

        /**
         * @param args the command line arguments
         */
        static void Main(string[] args)
        {
            if (args.Length < 8)
            {
                printUsage();
                return;
            }

            String colName = args[2];
            float valueC = float.Parse(args[3]);
            float valueM = float.Parse(args[4]);
            float valueY = float.Parse(args[5]);
            float valueK = float.Parse(args[6]);
            float tint = float.Parse(args[7]);
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    //ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            TextWriter errorWriter = Console.Error;
                            errorWriter.WriteLine("ページコンテナが空");
                            return;
                        }
                        using (PtlPage page = pages.get(0))     // ページの取得(先頭ページを取得)
                        using (PtlContent content = page.getContent())  // ページコンテントの取得
                        using (PtlRect outputRect = page.getViewBox())  // 出力矩形の設定。(ページの大きさを得るためにViewBoxを取得する)
                        using (PtlParamFont font = new PtlParamFont("ＭＳ ゴシック", 36, true, false, true))  // 出力テキストのフォント
                        using (PtlParamWriteString writeStringParam = new PtlParamWriteString())    // 文字描画のパラメータクラス
                        // 色表現用のクラスを作成
                        using (PtlColorSpaceSeparation colorSpace
                            = new PtlColorSpaceSeparation(colName, valueC, valueM, valueY, valueK))
                        // セパレーション色を表現したクラス
                        using (PtlColorSeparation colSeparation = new PtlColorSeparation(colorSpace, tint))
                        {
                            //文字の色指定
                            writeStringParam.setTextColor(colSeparation);

                            //フォントの指定（文字色を確認しやすくするため大きく描画する）
                            writeStringParam.setFont(font);
                            // 文字列出力 (中央に文字列を作成)
                            content.writeString(outputRect, PtlContent.ALIGN.ALIGN_CENTER,
                                                "writeString Test : set color", writeStringParam);
                        }
                    }
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                       "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }
    }
}
