﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：X.509公開鍵ファイルを用いてPDFを暗号化する

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace EncryptSetPubKeyWithX509
{
    internal class EncryptSetPubKeyWithX509
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.Write("usage: EncryptSetPubKeyWithX509 in-pdf-file out-pdf-file ");
            Console.WriteLine("暗号化種類 X509Pubkey-path 印刷権限種類 変更権限 コピー可否 "
                             + "アクセシビリティ有効の可否");
            Console.WriteLine("暗号化種類");
            Console.WriteLine("0 : 128 bit RC4  1 : 128 bit AES  2 : 256 bit AES");
            Console.WriteLine("印刷権限種類\n0 : 印刷不可  1 : 低解像度で可  2 : 高解像度で可");
            Console.WriteLine("変更権限種類");
            Console.WriteLine("0 : 許可しない");
            Console.WriteLine("1 : 注釈の作成、フォームフィールドの入力と既存の署名フィールドに署名を許可");
            Console.WriteLine("2 : ページの挿入、削除、回転を許可");
            Console.WriteLine("3 : フォームフィールドの入力と既存の署名フィールドに署名を許可");
            Console.WriteLine("4 : ページ抽出を除く全ての動作を許可");
            Console.WriteLine("コピー可否\n0 : コピーを許可しない 1 : コピーを許可する");
            Console.WriteLine("アクセシビリティの可否");
            Console.Write("（スクリーンリーダーデバイスのテキストアクセス）を");
            Console.WriteLine("有効にするか");
            Console.WriteLine("0 : 有効にしない 1 : 有効にする");
        }

        static void Main(string[] args)
        {
            if (args.Length < 8)
            {
                printUsage();
                return;
            }

            // コマンドライン引数の取得
            int kind;
            String permPrint = "";
            String permModify = "";
            bool permCopy;
            bool permAccessibility;

            try
            {
                kind = int.Parse(args[2]);
                switch (kind)
                {
                    case 0:
                    case 1:
                    case 2:
                        break;
                    default:
                        Console.WriteLine("暗号化種類には、0から2の数値を指定してください。");
                        printUsage(); // usageメッセージの表示
                        return;
                }
            }
            catch (FormatException numfe)
            {
                Console.WriteLine("ERROR!!暗号化種類の文字形式が正しくありません。");
                Console.WriteLine("暗号化種類には、0から2の数値を指定してください。");
                Console.WriteLine(numfe.StackTrace);
                printUsage(); // usageメッセージの表示
                return;
            }
            //印刷権限種類
            try
            {
                permPrint = setPrintForPermType2(args[4]);
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }
            //変更権限種類
            try
            {
                permModify = setModifyForPermType2(args[5]);
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }
            //コピー可否
            try
            {
                permCopy = readBoolArgs(args[6],
                                             "コピー可否は" +
                                               "0か1で指定してください。");
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }
            try
            {
                permAccessibility = readBoolArgs(args[7],
                                    "アクセシビリティ有効の可否は、0か1で指定してください。");
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlParamInput x509Pubkey = new PtlParamInput(args[3]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします
                    doc.load(inputFile);
                    // PDFにX.509形式の証明書を設定します
                    switch (kind)
                    {
                        case 0:
                            // 128 bit RC4
                            Console.WriteLine("128bit RC4で暗号化します。");
                            encryptX509Pubkey128RC4(doc, x509Pubkey, permPrint, permModify, permCopy, permAccessibility);
                            break;
                        case 1:
                            // 128 bit AES
                            Console.WriteLine("128bit AESで暗号化します。");
                            encryptX509Pubkey128AES(doc, x509Pubkey, permPrint, permModify, permCopy, permAccessibility);
                            break;
                        case 2:
                            // 256 bit AES
                            Console.WriteLine("256bit AESで暗号化します。");
                            encryptX509Pubkey256AES(doc, x509Pubkey, permPrint, permModify, permCopy, permAccessibility);
                            break;
                    }
                    // ファイルに保存します
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        private static void encryptX509Pubkey128RC4(PtlPDFDocument doc, PtlParamInput x509Pubkey,
                                                    String permPrint, String permModify,
                                                    bool permCopy, bool permAccessibility)
        {
            try
            {
                using (PtlEncryptPubKey128RC4 encPubkey128RC4 = new PtlEncryptPubKey128RC4())
                {
                    //128ビットRC4暗号化された公開キーセキュリティハンドラを表すクラス
                    encryptX509Pubkey(doc, x509Pubkey, encPubkey128RC4, permPrint, permModify, permCopy, permAccessibility);
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (IOException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void encryptX509Pubkey128AES(PtlPDFDocument doc, PtlParamInput x509Pubkey,
                                                    String permPrint, String permModify,
                                                    bool permCopy, bool permAccessibility)
        {
            try
            {
                using (PtlEncryptPubKey128AES encPubkey128AES = new PtlEncryptPubKey128AES())
                {
                    //128ビットAES暗号化された公開キーセキュリティハンドラを表すクラス
                    encryptX509Pubkey(doc, x509Pubkey, encPubkey128AES, permPrint, permModify, permCopy, permAccessibility);
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (IOException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void encryptX509Pubkey256AES(PtlPDFDocument doc, PtlParamInput x509Pubkey,
                                                    String permPrint, String permModify,
                                                    bool permCopy, bool permAccessibility)
        {
            try
            {
                using (PtlEncryptPubKey256AES encPubkey256AES = new PtlEncryptPubKey256AES())
                {
                    //256ビットAES暗号化された公開キーセキュリティハンドラを表すクラス
                    encryptX509Pubkey(doc, x509Pubkey, encPubkey256AES, permPrint, permModify, permCopy, permAccessibility);
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (IOException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void encryptX509Pubkey(PtlPDFDocument doc, PtlParamInput x509Pubkey,
                                              PtlEncryptPubKey encPubkey,
                                              String permPrint, String permModify,
                                              bool permCopy, bool permAccessibility)
        {
            try
            {
                using (PtlRecipients recipients = encPubkey.getRecipients())   //証明書受信者のコンテナを表したクラス
                using (PtlRecipient recipient = new PtlRecipient())            //証明書の受信者を表したクラス
                using (PtlEncryptPermissionPubKey permission = new PtlEncryptPermissionPubKey())
                {
                    //受信者クラスにX.509形式の証明書を設定
                    recipient.setX509(x509Pubkey);

                    //受信者クラスにユーザーアクセス許可フラグを設定
                    // 印刷権限の設定
                    // PERM_PRINT_NOT_ALLOWED : 許可しない
                    permission.setPrint((PtlEncryptPermissionType2.PERMISSION_PRINT)valuePrint(permPrint));

                    // 変更権限の設定
                    // PERM_MODIFY_NOT_ALLOWED : 許可しない
                    permission.setModify((PtlEncryptPermissionType2.PERMISSION_MODIFY)valueModify(permModify));

                    // テキスト、画像、その他の内容のコピーを有効にするかどうかの設定
                    // false : 有効にしない
                    permission.setCopy(permCopy);

                    // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの設定
                    // false : 有効にしない
                    permission.setAccessibility(permAccessibility);
                    //設定したユーザーアクセスフラグを設定する
                    recipient.setPermission(permission);

                    //受信者コンテナのクラスに受信者クラスを追加
                    recipients.append(recipient);

                    //以下は標準セキュリティハンドラと共通の設定
                    //すべてのコンテンツを暗号化する
                    encPubkey.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL);

                    // 設定したセキュリティハンドラを用いてPDFを暗号化
                    doc.setEncrypt(encPubkey);
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (IOException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        public static String setPrintForPermType2(String permPrintArg)
        {
            String permPrint = "PERM_PRINT_NOT_ALLOWED";
            try
            {
                switch (permPrintArg)
                {
                    case "0":
                        // PERM_MODIFY_NOT_ALLOWEDを用いるため、変更無し
                        break;
                    case "1":
                        permPrint = "PERM_PRINT_LOW";
                        break;
                    case "2":
                        permPrint = "PERM_PRINT_HIGH";
                        break;

                    default:
                        throw new ArgumentException("印刷権限種類には、0から2の数値を指定してください。");
                }

            }
            catch (ArgumentException ex)
            {
                throw ex;
            }
            return permPrint;
        }

        public static String setModifyForPermType2(String permModifyArg)
        {
            String permModify2 = "PERM_MODIFY_NOT_ALLOWED";
            try
            {
                switch (permModifyArg)
                {
                    case "0":
                        // PERM_MODIFY_NOT_ALLOWEDを用いるため、変更無し
                        break;
                    case "1":
                        permModify2 = "PERM_MODIFY_ANNOTANDFORM";
                        break;
                    case "2":
                        permModify2 = "PERM_MODIFY_ASSEMBLEDOC";
                        break;
                    case "3":
                        permModify2 = "PERM_MODIFY_FILLFORM";
                        break;
                    case "4":
                        permModify2 = "PERM_MODIFY_MODYFYDOC";
                        break;
                    default:
                        throw new ArgumentException("変更権限種類には、0から4の数値を指定してください。");
                }
            }
            catch (ArgumentException ex)
            {
                throw ex;
            }
            return permModify2;
        }

        /**
         * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
         * 
         * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
         * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
         * @return argsの数値を読み取った結果を戻す
         * @throws ArgumentException argsが0か1でなかった場合に発生。
         */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;
            try
            {
                // argsの読み取り
                switch (args)
                {
                    case "0":
                        trueOrFalse = false;
                        break;
                    case "1":
                        trueOrFalse = true;
                        break;
                    default:
                        throw new ArgumentException(errorMessage);
                }
            }
            catch (ArgumentException ex)
            {
                throw ex;
            }
            return trueOrFalse;
        }

        // 文字列を PtlEncryptPermissionType2.PERMISSION_PRINT の数値型として返す
        static int valuePrint(String printer)
        {
            if (printer.Equals(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_HIGH.ToString()))
            {
                return (int)PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_HIGH;
            }
            else if (printer.Equals(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_LOW.ToString()))
            {
                return (int)PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_LOW;
            }
            else if (printer.Equals(PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_NOT_ALLOWED.ToString()))
            {
                return (int)PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_NOT_ALLOWED;
            }
            else
            {
                throw new ArgumentException();
            }
        }

        // 文字列を PtlContent.ALIGN の数値型として返す
        static int valueModify(String modify)
        {
            if (modify.Equals(PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_ANNOTANDFORM.ToString()))
            {
                return (int)PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_ANNOTANDFORM;
            }
            else if (modify.Equals(PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_ASSEMBLEDOC.ToString()))
            {
                return (int)PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_ASSEMBLEDOC;
            }
            else if (modify.Equals(PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_FILLFORM.ToString()))
            {
                return (int)PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_FILLFORM;
            }
            else if (modify.Equals(PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_MODYFYDOC.ToString()))
            {
                return (int)PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_MODYFYDOC;
            }
            else if (modify.Equals(PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_NOT_ALLOWED.ToString()))
            {
                return (int)PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_NOT_ALLOWED;
            }
            else
            {
                throw new ArgumentException();
            }
        }
    }
}
