﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：座標ベースでのテキスト抽出の際に同じテキストが
        どの程度重なり合っていたら除外するかを指定します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace ExtTextSetSameTextOmitRatio
{
    internal class ExtTextSetSameTextOmitRatio
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: ExtTextSetSameTextOmitRatio"
                             + " in-pdf-file out-text-file"
                             + " page-to-extract sameText-omit-ratio");
            Console.WriteLine("sameText-omit-ratio:");
            Console.WriteLine("同じ文字が重なり合っていた場合にどの程度重なり合っていたら除外するかを小数で指定");
        }

        static void Main(string[] args)
        {
            if (args.Length < 4)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の読み取り・判定
            // 出力PDFの名前はあとで渡すためにString型で保存する。
            String outputTextURI = args[1];
            int pageToExtract = int.Parse(args[2]);
            float sameTextOmitRatio = float.Parse(args[3]);
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロード
                    doc.load(inputFile);
                    //ページ数を取得
                    int wholePageNum = doc.getPageCount();
                    //pageToExtractのエラー処理
                    if (wholePageNum < pageToExtract)
                    {
                        Console.WriteLine("ERROR: page-to-extractはPDFの総ページ数より" +
                                           "小さい値を指定してください。");
                        Console.WriteLine("総ページ数：" + wholePageNum);
                        printUsage();
                        return;
                    }
                    //ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ERROR : ページコンテナが空");
                            return;
                        }
                        // ページの取得(パラメータindexは0が先頭のため1を引く)
                        using (PtlPage page = pages.get(pageToExtract - 1))
                        // ページコンテントの取得
                        using (PtlContent content = page.getContent())
                        // 文字抽出のパラメータクラス。
                        using (PtlParamExtractText paramExtractText = new PtlParamExtractText())
                        {
                            // 同じテキストがどのぐらい重なっていたら取り除くかを指定する。
                            paramExtractText.setSameTextOmitRatio(sameTextOmitRatio);

                            // 抽出するテキストを座標順に並べるよう指定
                            //（setSameTextOmitRatioのためには指定が必要）
                            paramExtractText.setTextType(PtlParamExtractText.TEXT_TYPE.TEXT_SORT);

                            // 文字列抽出
                            String TextFromPdf = content.extractText(paramExtractText);
                            // 文字列の出力
                            //outputTextFile(outputTextURI, TextFromPdf);
                            File.WriteAllText(outputTextURI, TextFromPdf);
                            Console.WriteLine(TextFromPdf);
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }
    }
}
