﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：テキスト抽出の際に含まれた特殊文字のうち、
        指定したものを削除します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;
using System.Text;

namespace ExtTextWithRemovingSoftHyphen
{
    internal class ExtTextWithRemovingSoftHyphen
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: ExtTextWithRemovingSoftHyphen in-pdf-file"
                             + " out-text-file page-to-extract rem-soft-hyphen");
            Console.WriteLine("--以下の文字に対して、[0:削除しない  1:削除する]を選択--");
            Console.WriteLine("rem-soft-hyphen : ソフトハイフン");
        }

        static void Main(string[] args)
        {
            if (args.Length < 4)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の読み取り・判定
            // 出力PDFの名前はあとで渡すためにString型で保存する。
            String outputTextURI = args[1];
            int pageToExtract = int.Parse(args[2]);
            bool remSoftHyphen = false;
            int unicodeCharFlag = 0;

            //各種削除フラグの読み取り判定
            //rem-soft-hyphen
            try
            {
                remSoftHyphen = readBoolArgs(args[3], "rem-soft-hyphenは0か1で指定してください。");
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }

            //各フラグの論理和をunicodeCharFlagに設定
            if (remSoftHyphen)
            {
                unicodeCharFlag = unicodeCharFlag | (int)PtlParamExtractText.UNI_FLAG.UNI_SOFT_HYPHEN;//.AHEXTRACTTEXT_UNI_SOFT_HYPHEN;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロード
                    doc.load(inputFile);
                    try
                    {
                        using (PtlPages pages = doc.getPages()) //ページコンテナの取得
                        {
                            // ページコンテナが空かどうか
                            if (pages.isEmpty())
                            {
                                Console.WriteLine("ERROR : ページコンテナが空");
                                throw new Exception("ERROR : ページコンテナが空");
                            }
                            //ページ数を取得
                            int wholePageNum = doc.getPageCount();

                            //pageToExtractが0ならすべてのページを処理する
                            if (pageToExtract == 0)
                            {
                                StringBuilder wholeTextFromPdf = new StringBuilder();
                                // ページの取得(パラメータindexは0が先頭のため1を引く)
                                for (int i = 0; i < wholePageNum; i++)
                                {
                                    using (PtlPage page = pages.get(i))
                                    using (PtlContent content = page.getContent())
                                    { // ページコンテントの取得
                                        Console.WriteLine((i + 1) + "ページ目のテキストを抽出します。");
                                        wholeTextFromPdf.Append(extractTextSetRemoveUnicodeFlag(unicodeCharFlag, content));
                                    }
                                }
                                File.WriteAllText(outputTextURI, wholeTextFromPdf.ToString());
                            }
                            else
                            {
                                if (wholePageNum < pageToExtract)
                                { //pageToExtractのエラー処理
                                    Console.WriteLine("ERROR: page-to-extractはPDFの総ページ数より" +
                                                       "小さい値を指定してください。");
                                    Console.WriteLine("総ページ数：" + wholePageNum);
                                    printUsage();
                                    throw new Exception("ERROR: page-to-extractはPDFの総ページ数より" +
                                                    "小さい値を指定してください。");
                                }
                                using (PtlPage page = pages.get(pageToExtract - 1)) //指定したページを取得する
                                using (PtlContent content = page.getContent())
                                {
                                    Console.WriteLine(pageToExtract + "ページ目のテキストを抽出します。");
                                    File.WriteAllText(outputTextURI, extractTextSetRemoveUnicodeFlag(unicodeCharFlag, content));
                                }
                            }
                        }
                    }
                    catch (PtlException pex)
                    {
                        Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        private static String extractTextSetRemoveUnicodeFlag(int unicodeCharFlag, PtlContent content)
        {
            try
            {
                using (PtlParamExtractText paramExtractText = new PtlParamExtractText())
                { // 文字抽出のパラメータクラス。
                  //setUnicodeToRemoveをパラメータに設定
                    paramExtractText.setUnicodeToRemove((ushort)unicodeCharFlag);
                    // 文字列抽出
                    String textFromPdf = content.extractText(paramExtractText);
                    Console.WriteLine(textFromPdf);

                    // 抽出した文字列を返す
                    return textFromPdf;
                }
            }
            catch (IOException ex)
            {
                throw ex;
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
            * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
            * 
            * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
            * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
            * @return argsの数値を読み取った結果を戻す
            * @throws ArgumentException argsが0か1でなかった場合に発生。
            */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;
            try
            {
                // argsの読み取り
                switch (args)
                {
                    case "0":
                        trueOrFalse = false;
                        break;
                    case "1":
                        trueOrFalse = true;
                        break;
                    default:
                        throw new ArgumentException(errorMessage);
                }
            }
            catch (ArgumentException ex)
            {
                throw ex;
            }
            return trueOrFalse;
        }
    }
}
