﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要： JPEG画像の書き出し時にパススルー設定をするかどうかを選択します。
        指定されたインデックス番号の画像エレメントを取得し、
  　　　画像エレメントからファイルを書き出します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace ExtractImageSetPassthrough
{
    internal class ExtractImageSetPassthrough
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: ExtractImageSetPassthrough in-pdf-file out-image-body" +
                               " page-to-extract image-index-num set-passthrough");
            Console.WriteLine("set-passthrough : ");
            Console.WriteLine("0 : JPEG圧縮する  1 : JPEGは加工せずにそのまま出力する");
        }

        static void Main(string[] args)
        {
            if (args.Length < 5)
            {
                printUsage();
                return;
            }

            // コマンドライン引数の取得
            int pageToExtract = int.Parse(args[2]);
            int imageIndexNum = int.Parse(args[3]);
            bool setPassthrough = false;
            try
            {
                setPassthrough = readBoolArgs(args[4], "set-passthroughは 0か1で指定してください。");
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            Console.WriteLine("ページコンテナが空");
                            return;
                        }

                        // それ以外であれば指定ページ番号を検索。
                        int wholePageNum = doc.getPageCount();
                        //pageToSearchのエラー処理
                        if (wholePageNum < pageToExtract)
                        {
                            Console.WriteLine("ERROR: page-to-extractはPDFの総ページ数より" +
                                               "小さい値を指定してください。");
                            Console.WriteLine("総ページ数：" + wholePageNum);
                            printUsage();
                            return;
                        }
                        // ページの取得(index番号は0が先頭のため1を引く)
                        using (PtlPage page = pages.get(pageToExtract - 1))
                        {
                            // 画像抽出
                            extractImageSetPassthrough(page, args[1], imageIndexNum, setPassthrough);
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void extractImageSetPassthrough(PtlPage page, String imagefileName,
                                              int imageIndexNum, bool setPassthrough)
        {
            // ページコンテント・画像エレメントの取得
            try
            {
                using (PtlContent content = page.getContent())
                using (PtlEditElements elems = content.getEditElements(PtlContent.GET_ELEMENT.GET_IMAGE))
                {
                    int numElems = elems.getCount();
                    if (numElems == 0)
                    {
                        Console.WriteLine("指定されたページに画像エレメントがありませんでした。");
                        return; // 画像エレメントが無い場合はreturnする
                    }
                    //imageIndexNumのエラー処理
                    if (numElems < imageIndexNum)
                    {
                        Console.WriteLine("ERROR: image-index-numは対象ページの総画像数より" +
                                           "小さい値を指定してください。");
                        Console.WriteLine("総画像数：" + numElems);
                        printUsage();
                        return;
                    }
                    // 画像エレメントの取得
                    using (PtlEditElement elem = elems.get(imageIndexNum - 1))
                    {
                        PtlEditElement.ELEMENT_TYPE type = elem.getType();
                        switch (type)
                        {
                            case PtlEditElement.ELEMENT_TYPE.TYPE_IMAGE:
                                PtlEditImage elemImage = (PtlEditImage)elem;
                                elemImage.setPathThrough(setPassthrough);
                                using (PtlParamOutput outputImage = new PtlParamOutput(imagefileName))
                                {
                                    // 画像出力(パススルー設定を確かめるため、FORMAT_JPEGで出力する。)
                                    elemImage.writeFile(outputImage, PtlEditImage.OUTPUT_FORMAT.FORMAT_JPEG);
                                }
                                break;
                            default:
                                Console.WriteLine("指定されたエレメントは画像ではありません。処理を中断します。");
                                break;
                        }
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
                //throws PtlException, Exception, Error 
            }
            catch (Exception ex)
            {
                throw ex;
                //throws PtlException, Exception, Error 
            }
        }

        /**
         * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
         * 
         * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
         * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
         * @return argsの数値を読み取った結果を戻す
         * @throws ArgumentException argsが0か1でなかった場合に発生。
         */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;
            try
            {
                // argsの読み取り
                switch (args)
                {
                    case "0":
                        trueOrFalse = false;
                        break;
                    case "1":
                        trueOrFalse = true;
                        break;
                    default:
                        throw new ArgumentException(errorMessage);
                }
            }
            catch (ArgumentException ex)
            {
                throw ex;
                //throws PtlException, Exception, Error 
            }
            return trueOrFalse;
        }
    }
}
