﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：入力PDFをPDF/A-1bまたはPDF/A-2bに変換する。
        変換の際にリニアライズして保存するかを選択する。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace FixUpPDFASetSaveOption
{
    internal class FixUpPDFASetSaveOption
    {
        // そのクラスのusageを表示する関数
        public static void printUsage()
        {
            Console.Write("usage: FixUpPDFA in-pdf-file out-pdf-file");
            Console.WriteLine(" PDF/Aの種類 保存時のオプション\n");
            Console.WriteLine("PDF/Aの種類");
            Console.WriteLine("1 : PDF/A-1b  2 : PDF/A-2b");
            Console.WriteLine("保存時のオプション");
            Console.WriteLine("0 : デフォルト  1 : リニアライズする");
        }

        static void Main(string[] args)
        {
            if (args.Length < 4)
            {
                printUsage();
                return;
            }

            int kind;
            try
            {
                kind = int.Parse(args[2]);
                switch (kind)
                {
                    case 1:
                    case 2:
                        break;
                    default:
                        Console.WriteLine("PDF-Aの種類には、1または2の数値を指定してください。");
                        printUsage(); // usageメッセージの表示
                        return;
                }
            }
            catch (FormatException)
            {
                Console.WriteLine("PDF-Aの種類には、1または2の数値を指定してください。");
                printUsage(); // usageメッセージの表示
                return;
            }

            bool setLinearize;
            try
            {
                setLinearize = readBoolArgs(args[3],
                                      "保存時のオプションには、0または1の数値を指定してください。");
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }

            try
            {
                // コマンドライン引数の取得
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFFixUp fixUp = new PtlPDFFixUp())
                {
                    if (setLinearize)
                    {
                        // リニアライズする
                        fixUp.setSaveOption(PtlPDFFixUp.SAVE_OPTION.SAVE_LINEARIZE);
                    }
                    else
                    {
                        //デフォルト
                        fixUp.setSaveOption(PtlPDFFixUp.SAVE_OPTION.SAVE_RECONSTRUCT);
                    }

                    // PDFFixUPでinputFileをPDF/Aにする。
                    switch (kind)
                    {
                        case 1: //PDF/A-1b
                            fixUp.fixUpPDFA(PtlPDFFixUp.PDFA_TYPE.PDFA_1B, inputFile);
                            break;
                        case 2: // PDF/A-2b
                            fixUp.fixUpPDFA(PtlPDFFixUp.PDFA_TYPE.PDFA_2B, inputFile);
                            break;
                    }

                    //変換エラー発生チェック
                    using (PtlPDFFixUpErrors fixUpErrors = fixUp.getErrors())
                    {
                        int numErrors = fixUpErrors.getCount();
                        Console.WriteLine("変換時の総エラー数 : " + numErrors);

                        if (numErrors == 0)
                        {
                            Console.WriteLine("エラーは発生しませんでした。");
                        }
                        else
                        {
                            for (int i = 0; i < numErrors; i++)
                            {

                                using (PtlPDFFixUpError thisFixUpError = fixUpErrors.get(i))
                                {
                                    Console.WriteLine("エラーその" + i + "：" + thisFixUpError.getErrorMessageJP());
                                }
                            }
                        }
                    }

                    // ファイルに保存します。
                    fixUp.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        /**
         * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
         * 
         * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
         * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
         * @return argsの数値を読み取った結果を戻す
         * @throws ArgumentException argsが0か1でなかった場合に発生。
         */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;
            try
            {
                // argsの読み取り
                switch (args)
                {
                    case "0":
                        trueOrFalse = false;
                        break;
                    case "1":
                        trueOrFalse = true;
                        break;
                    default:
                        throw new ArgumentException(errorMessage);
                }
            }
            catch (ArgumentException ex)
            {
                throw ex;
            }
            return trueOrFalse;
        }
    }
}
