﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：文書情報の取得

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace GetDocInfoWithFontInfo
{
    internal class GetDocInfoWithFontInfo
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: GetDocInfoWithFontInfo in-pdf-file");
        }

        static void Main(string[] args)
        {
            if (args.Length < 1)
            {
                printUsage();
                return;
            }
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロード
                    doc.load(inputFile);

                    using (PtlDocProperty docProperty = doc.getDocProperty()) //PDFの文書プロパティ
                    using (PtlDocInfo docinf = docProperty.getDocInfo())    //PDFの文書情報
                                                                            // PDFのカスタムプロパティ
                    using (PtlCustomProperties customProperties = docProperty.getCustomProperties())
                    //フォント情報を扱うコンテナ
                    using (PtlFontInfos fontInfos = docProperty.getFontInfos())
                    {
                        //フォント情報を取得
                        getDocFontInfo(fontInfos);

                        // タイトル取得
                        Console.WriteLine("Title : " + docinf.getTitle());
                        // 著者取得
                        Console.WriteLine("Author : " + docinf.getAuthor());
                        // サブジェクト取得
                        Console.WriteLine("Subject : " + docinf.getSubject());
                        // キーワード取得
                        Console.WriteLine("Keywords : " + docinf.getKeywords());
                        // クリエータ取得
                        Console.WriteLine("Creator : " + docinf.getCreator());
                        // プロデューサ取得
                        Console.WriteLine("Producer : " + docinf.getProducer());
                        // 作成日付を取得
                        using (PtlDate dateCreate = docinf.getCreationDate())
                        {
                            Console.WriteLine("CreationDate : "
                                + dateCreate.getYear()
                                + "/" + dateCreate.getMonth()
                                + "/" + dateCreate.getDay()
                                + " " + dateCreate.getHour()
                                + ":" + dateCreate.getMin()
                                + ":" + dateCreate.getSec());
                        }
                        // 更新日付を取得
                        using (PtlDate dateMod = docinf.getModDate())
                        {
                            Console.WriteLine("ModDate : "
                                + dateMod.getYear()
                                + "/" + dateMod.getMonth()
                                + "/" + dateMod.getDay()
                                + " " + dateMod.getHour()
                                + ":" + dateMod.getMin()
                                + ":" + dateMod.getSec());
                        }
                        // カスタムプロパティが空かどうか判定
                        Console.WriteLine("");
                        if (customProperties.isEmpty())
                        {
                            //空の場合はそう表示する
                            Console.WriteLine("このPDFにカスタムプロパティは設定されていません。");
                        }
                        else
                        {  // カスタムプロパティが空でない場合に内容を取得する
                            Console.WriteLine("このPDFのカスタムプロパティを表示します。");
                            int propertiesNum = customProperties.getCount();
                            for (int i = 0; i < propertiesNum; i++)
                            {
                                Console.WriteLine("Custom Property No." + (i + 1) + ":");
                                using (PtlCustomProperty customProperty = customProperties.get(i))
                                {
                                    Console.WriteLine("名前 : " + customProperty.getName());
                                    Console.WriteLine("値 : " + customProperty.getValue());
                                    Console.WriteLine("");
                                }
                            }
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        /**
         * ドキュメントが持つフォント情報を全て表示する関数。
         * @param fontInfos
         * @throws PtlException
         * @throws Exception
         */
        private static void getDocFontInfo(PtlFontInfos fontInfos)
        {
            try
            {
                Console.WriteLine("Font : ");
                //フォント情報の有無を確認する
                if (fontInfos.isEmpty())
                {
                    Console.WriteLine("  このドキュメントにフォント情報はありません。");
                }
                else
                {
                    //フォント情報がある場合の処理を行う
                    int fontNum = fontInfos.getCount();
                    Console.WriteLine("  フォント総数 : " + fontNum);
                    for (int i = 0; i < fontNum; i++)
                    {
                        using (PtlFontInfo fontInfo = fontInfos.get(i))
                        {
                            Console.WriteLine("  --Font index No: " + i);
                            Console.WriteLine("  フォント名：" + fontInfo.getFontName());
                            Console.WriteLine("  フォントタイプ：" + fontInfo.getFontType());
                            Console.WriteLine("  エンコーディング名：" + fontInfo.getEncodingName());
                            Console.WriteLine("  エンコーディングタイプ：" + fontInfo.getEncodingType());
                            if (fontInfo.isEmbedded())
                            {
                                Console.WriteLine("  フォント埋め込み：あり");
                            }
                            else
                            {
                                Console.WriteLine("  フォント埋め込み：なし");
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
    }
}
