﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：セキュリティ情報の取得
  PDF Tool API V7.0以降ではセキュリティ情報に関して標準セキュリティハンドラだけでなく
  公開キーセキュリティハンドラの情報も扱えます。
  このサンプルプログラムでは双方の情報を取得します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace GetEncryptInfoPubkeyPKCS12
{
    internal class GetEncryptInfoPubkeyPKCS12
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: GetEncryptInfoPubkeyPKCS12 in-pdf-file "
                    + "PKCS#12-path [pkcs#12-password]");
        }

        static void Main(string[] args)
        {
            if (args.Length < 2)
            {
                printUsage();
                return;
            }
            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamInput pkcs12Pubkey = new PtlParamInput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    doc.setPassword(args[2]);
                    doc.setPKCS12(pkcs12Pubkey);
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // 暗号化の取得
                    if (doc.isEncrypted())
                    {
                        // 暗号化情報の取得
                        showStdAndPermEncInfo(doc);
                    }
                    else
                    {
                        Console.WriteLine("暗号化されていないファイルです");
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void showStdAndPermEncInfo(PtlPDFDocument doc)
        {
            try
            {
                //全タイプのセキュリティハンドラで共通のデータを取得
                using (PtlEncrypt encrypt = doc.getEncrypt())
                {
                    // キー長の取得
                    Console.WriteLine("暗号化のキー長 : " + encrypt.getKeyLength());

                    // 暗号化する文書コンポーネント取得
                    switch (encrypt.getEncryptComponent())
                    {
                        case PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL:
                            Console.WriteLine("文書の全てのコンテンツを暗号化");
                            break;
                        case PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_EXCEPT_METADATA:
                            Console.WriteLine("文書のメタデータを除く全てのコンテンツを暗号化");
                            break;
                        case PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ONLY_ATTACHED_FILE:
                            Console.WriteLine("添付ファイルのみを暗号化(AES暗号化のみで有効)");
                            break;
                    }

                    // 暗号化のフィルター取得
                    // FILTER_STANDARD : 標準セキュリティハンドラ */
                    if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_STANDARD)
                    {
                        //標準セキュリティハンドラの情報取得
                        //PtlEncryptStandardにダウンキャストして情報を渡す
                        getEncryptStandardInfo((PtlEncryptStandard)encrypt);
                    }
                    // FILTER_PUBKEY : 公開キーセキュリティハンドラ */
                    else if (encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_PUBKEY)
                    {
                        //公開キーセキュリティハンドラの情報取得
                        //PtlEncryptPubkeyにダウンキャストして情報を渡す
                        getEncryptPubkeyInfo((PtlEncryptPubKey)encrypt);
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * 公開キーセキュリティハンドラの情報を取得するメソッド
         * @param encryptPubkey 公開キーセキュリティハンドラを表すクラス
         * @throws PtlException
         * @throws Exception
         * @throws Error 
         */
        private static void getEncryptPubkeyInfo(PtlEncryptPubKey encryptPubkey)
        {
            try
            {
                Console.WriteLine("---セキュリティハンドラのタイプは公開キーセキュリティハンドラです。---");
                // getFilterType()で判定済みなので PtlEncryptPubKey にダウンキャストする。

                // 公開キーセキュリティハンドラのメソッド取得
                switch (encryptPubkey.getMethod())
                {
                    case PtlEncryptPubKey.PUBSEC_METHOD_TYPE.METHOD_RC4:
                        Console.WriteLine("公開キーセキュリティハンドラのメソッド : RC4");
                        break;
                    case PtlEncryptPubKey.PUBSEC_METHOD_TYPE.METHOD_AES:
                        Console.WriteLine("公開キーセキュリティハンドラのメソッド : AES");
                        break;
                }

                // ユーザアクセス許可フラグの取得
                // PtlEncryptPermissionPubKeyを取得する
                Console.WriteLine("--ユーザアクセス許可フラグの情報を取得します--");
                using (PtlEncryptPermissionPubKey permsPubkey = (PtlEncryptPermissionPubKey)encryptPubkey.getPermission())
                {
                    if (permsPubkey.hasFullPermission())
                    {
                        Console.WriteLine("このPDFではすべての権限が許可されています");
                    }
                    else
                    {
                        Console.WriteLine("このPDFでは個別に権限の許可・許可しないが設定されます");
                    }
                    // 印刷権限取得
                    switch (permsPubkey.getPrint())
                    {
                        case PtlEncryptPermissionPubKey.PERMISSION_PRINT.PERM_PRINT_NOT_ALLOWED:
                            Console.WriteLine("印刷権限 : 許可しない");
                            break;
                        case PtlEncryptPermissionPubKey.PERMISSION_PRINT.PERM_PRINT_LOW:
                            Console.WriteLine("印刷権限 : 低解像度");
                            break;
                        case PtlEncryptPermissionPubKey.PERMISSION_PRINT.PERM_PRINT_HIGH:
                            Console.WriteLine("印刷権限 : 高解像度");
                            break;
                    }
                    // 変更権限取得
                    switch (permsPubkey.getModify())
                    {
                        case PtlEncryptPermissionPubKey.PERMISSION_MODIFY.PERM_MODIFY_NOT_ALLOWED:
                            Console.WriteLine("変更権限 : 許可しない");
                            break;
                        case PtlEncryptPermissionPubKey.PERMISSION_MODIFY.PERM_MODIFY_ASSEMBLEDOC:
                            Console.WriteLine("変更権限 : ページの挿入、削除、回転");
                            break;
                        case PtlEncryptPermissionPubKey.PERMISSION_MODIFY.PERM_MODIFY_FILLFORM:
                            Console.WriteLine("変更権限 : フォームフィールドの入力と既存の署名フィールドに署名");
                            break;
                        case PtlEncryptPermissionPubKey.PERMISSION_MODIFY.PERM_MODIFY_ANNOTANDFORM:
                            Console.WriteLine("変更権限 : 注釈の作成、フォームフィールドの入力と既存の署名フィールドに署名");
                            break;
                        case PtlEncryptPermissionPubKey.PERMISSION_MODIFY.PERM_MODIFY_MODYFYDOC:
                            Console.WriteLine("変更権限 : ページ抽出を除く全ての動作");
                            break;
                    }
                    // テキスト、画像、その他の内容のコピーを有効にするかどうかの取得
                    if (permsPubkey.getCopy())
                    {
                        Console.WriteLine("テキスト、画像、その他の内容のコピーを有効にする");
                    }
                    else
                    {
                        Console.WriteLine("テキスト、画像、その他の内容のコピーを有効にしない");
                    }
                    // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの取得
                    if (permsPubkey.getAccessibility())
                    {
                        Console.WriteLine("スクリーンリーダーデバイスのテキストアクセスを有効にする");
                    }
                    else
                    {
                        Console.WriteLine("スクリーンリーダーデバイスのテキストアクセスを有効にしない");
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * 標準セキュリティハンドラの情報を取得するメソッド
         * @param encryptStandard 標準セキュリティハンドラを表すクラス
         * @throws PtlException
         * @throws Exception
         */
        private static void getEncryptStandardInfo(PtlEncryptStandard encryptStandard)
        {
            try
            {
                Console.WriteLine("---セキュリティハンドラのタイプは標準セキュリティハンドラです。---");
                // getFilterType()に判定済みなので PtlEncryptStandard にダウンキャストする。

                // 標準セキュリティハンドラのメソッド取得
                switch (encryptStandard.getMethod())
                {
                    case PtlEncryptStandard.STANDARD_METHOD_TYPE.METHOD_RC4:
                        Console.WriteLine("標準セキュリティハンドラのメソッド : RC4");
                        break;
                    case PtlEncryptStandard.STANDARD_METHOD_TYPE.METHOD_AES:
                        Console.WriteLine("標準セキュリティハンドラのメソッド : AES");
                        break;
                }

                // ユーザアクセス許可フラグの取得
                Console.WriteLine("--ユーザアクセス許可フラグの情報を取得します--");
                using (PtlEncryptPermission perms = encryptStandard.getPermission())
                {
                    // タイプ取得
                    switch (perms.getType())
                    {
                        case PtlEncryptPermission.PERM_TYPE.PERM_TYPE1:
                            {
                                Console.WriteLine("標準セキュリティハンドラのリビジョン : 2");
                                // getType()で判定されたのでpermsをダウンキャストできる。
                                PtlEncryptPermissionType1 perms1 = (PtlEncryptPermissionType1)perms;
                                // 印刷権限取得
                                if (perms1.getPrint())
                                {
                                    Console.WriteLine("印刷を許可する");
                                }
                                else
                                {
                                    Console.WriteLine("印刷を許可しない");
                                }

                                // 変更権限取得
                                switch (perms1.getModify())
                                {
                                    case PtlEncryptPermissionType1.PERMISSION_MODIFY.PERM_MODIFY_NOT_ALLOWED:
                                        Console.WriteLine("変更権限 : 許可しない");
                                        break;
                                    case PtlEncryptPermissionType1.PERMISSION_MODIFY.PERM_MODIFY_ANNOTANDFORM:
                                        Console.WriteLine("変更権限 : 注釈の作成、フォームフィールドの入力と既存の署名フィールドに署名");
                                        break;
                                    case PtlEncryptPermissionType1.PERMISSION_MODIFY.PERM_MODIFY_ASSEMBLEANDFORM:
                                        Console.WriteLine("変更権限 : ページレイアウト、フォームフィールドの入力と既存の署名フィールドに署名");
                                        break;
                                    case PtlEncryptPermissionType1.PERMISSION_MODIFY.PERM_MODIFY_MODYFYDOC:
                                        Console.WriteLine("変更権限 : ページ抽出を除く全ての動作");
                                        break;
                                }
                                // テキスト、画像、その他の内容のコピーおよびアクセシビリティを有効にするかどうかの取得
                                if (perms1.getCopy())
                                {
                                    Console.WriteLine("テキスト、画像、その他の内容のコピーおよびアクセシビリティを有効にする");
                                }
                                else
                                {
                                    Console.WriteLine("テキスト、画像、その他の内容のコピーおよびアクセシビリティを有効にしない");
                                }
                            }
                            break;

                        case PtlEncryptPermission.PERM_TYPE.PERM_TYPE2:
                            {
                                Console.WriteLine("標準セキュリティハンドラのリビジョン : 3または4");
                                // getType()で判定されたのでpermsをダウンキャストできる。
                                PtlEncryptPermissionType2 perms2 = (PtlEncryptPermissionType2)perms;
                                // 印刷権限取得
                                switch (perms2.getPrint())
                                {
                                    case PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_NOT_ALLOWED:
                                        Console.WriteLine("印刷権限 : 許可しない");
                                        break;
                                    case PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_LOW:
                                        Console.WriteLine("印刷権限 : 低解像度");
                                        break;
                                    case PtlEncryptPermissionType2.PERMISSION_PRINT.PERM_PRINT_HIGH:
                                        Console.WriteLine("印刷権限 : 高解像度");
                                        break;
                                }
                                // 変更権限取得
                                switch (perms2.getModify())
                                {
                                    case PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_NOT_ALLOWED:
                                        Console.WriteLine("変更権限 : 許可しない");
                                        break;
                                    case PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_ASSEMBLEDOC:
                                        Console.WriteLine("変更権限 : ページの挿入、削除、回転");
                                        break;
                                    case PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_FILLFORM:
                                        Console.WriteLine("変更権限 : フォームフィールドの入力と既存の署名フィールドに署名");
                                        break;
                                    case PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_ANNOTANDFORM:
                                        Console.WriteLine("変更権限 : 注釈の作成、フォームフィールドの入力と既存の署名フィールドに署名");
                                        break;
                                    case PtlEncryptPermissionType2.PERMISSION_MODIFY.PERM_MODIFY_MODYFYDOC:
                                        Console.WriteLine("変更権限 : ページ抽出を除く全ての動作");
                                        break;
                                }
                                // テキスト、画像、その他の内容のコピーを有効にするかどうかの取得
                                if (perms2.getCopy())
                                {
                                    Console.WriteLine("テキスト、画像、その他の内容のコピーを有効にする");
                                }
                                else
                                {
                                    Console.WriteLine("テキスト、画像、その他の内容のコピーを有効にしない");
                                }
                                // スクリーンリーダーデバイスのテキストアクセスを有効にするかどうかの取得
                                if (perms2.getAccessibility())
                                {
                                    Console.WriteLine("スクリーンリーダーデバイスのテキストアクセスを有効にする");
                                }
                                else
                                {
                                    Console.WriteLine("スクリーンリーダーデバイスのテキストアクセスを有効にしない");
                                }
                            }
                            break;
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
    }
}
