﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：入力PDFに含まれている全てのページの全ての画像について、
        画像に設定されているマスク情報を抜き出します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace GetImageMaskType
{
    internal class GetImageMaskType
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: GetImageMaskType in-pdf-file");
        }

        static void Main(string[] args)
        {
            if (args.Length < 1)
            {
                printUsage();
                return;
            }

            try {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    //ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                      // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            TextWriter errorWriter = Console.Error;
                            errorWriter.WriteLine("入力PDFのページコンテナが空");
                            return;
                        }
                        // それ以外であれば指定ページ番号を検索。
                        int wholePageNum = doc.getPageCount();
                        for (int i = 0; i < wholePageNum; i++)
                        {
                            // ページの取得(index番号は0が先頭のため1を引く)
                            using (PtlPage page = pages.get(i))
                            {
                                Console.WriteLine("このPDFの" +
                                                i + "ページ目の画像の情報を読み取ります。");
                                // 画像情報の読み取り
                                getImageMaskType(page);
                            }
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void getImageMaskType(PtlPage page)
        {
            try {
                // ページコンテント・画像エレメントの取得
                using (PtlContent content = page.getContent())
                using (PtlEditElements elems = content.getEditElements(PtlContent.GET_ELEMENT.GET_IMAGE))
                {
                    int numElems = elems.getCount();
                    if (numElems == 0) {
                        Console.WriteLine("GetImageMaskType 指定されたページに画像エレメントがありませんでした。");
                        return; // 画像エレメントが無い場合はreturnする
                    }
                    // 画像エレメントの取得
                    for (int i = 0; i < numElems; i++) {
                        Console.WriteLine("画像のインデックス番号：" + i);

                        //各エレメントの処理
                        using (PtlEditElement elem = elems.get(i))
                        {
                            bool isImage = false;
                            //エレメントが画像であるかの判定
                            switch (elem.getType()) {
                                case PtlEditElement.ELEMENT_TYPE.TYPE_IMAGE:
                                    isImage = true;
                                    break;
                                default:
                                    Console.WriteLine("指定されたエレメントは画像ではありません。");
                                    break;
                            }
                            if (isImage)
                            {
                                using (PtlEditImage elemImage = (PtlEditImage)elem)
                                {
                                    switch (elemImage.getMaskType())
                                    {
                                        case PtlEditImage.MASK_TYPE.MASK_COLORKEY:
                                            Console.WriteLine("この画像にはカラーキーマスクが設定されています。");
                                            break;
                                        case PtlEditImage.MASK_TYPE.MASK_EXPLICIT:
                                            Console.WriteLine("この画像には明示マスクが設定されています。");
                                            break;
                                        case PtlEditImage.MASK_TYPE.MASK_NONE:
                                            Console.WriteLine("この画像にはマスクが設定されていません。");
                                            break;
                                        case PtlEditImage.MASK_TYPE.MASK_SOFT:
                                            Console.WriteLine("この画像にはソフトマスクが設定されています。");
                                            break;
                                        case PtlEditImage.MASK_TYPE.MASK_STENCIL:
                                            Console.WriteLine("この画像にはステンシルマスクが設定されています。");
                                            break;
                                        //どのマスクタイプでもなかった場合の処理
                                        default:
                                            Console.WriteLine("マスクのタイプ情報を読み取れませんでした。");
                                            break;
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
    }
}
