﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：Contentに含まれるFormXOnjectエレメントの情報を表示します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace GetPtlEditFormXObjectInfo
{
    internal class GetPtlEditFormXObjectInfo
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: GetPtlEditFormXObjectInfo in-pdf-file");
        }

        static void Main(string[] args)
        {
            if (args.Length < 1)
            {
                printUsage();
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    //ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            TextWriter errorWriter = Console.Error;
                            errorWriter.WriteLine("入力PDFのページコンテナが空");
                            return;
                        }

                        // ページ数の取得
                        for (int i = 0; i < pages.getCount(); i++)
                        {
                            Console.WriteLine("-----------------");
                            Console.WriteLine(i + "ページ目の情報を表示します。");
                            using (PtlPage page = pages.get(i))
                            {
                                getFormXObjectInfoInPage(page);
                            }
                            Console.WriteLine("-----------------");
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        private static void getFormXObjectInfoInPage(PtlPage page)
        {
            try
            {
                using (PtlContent content = page.getContent())// 挿入先ページコンテントの取得
                using (PtlEditElements elems = content.getEditElements())
                {
                    int numOfElements = elems.getCount();
                    Console.WriteLine("このページに含まれるエレメントの数は" + numOfElements + "です。");
                    for (int i = 0; i < numOfElements; i++)
                    {
                        using (PtlEditElement element = elems.get(i))
                        {
                            if (element.getType() == PtlEditElement.ELEMENT_TYPE.TYPE_FORMXOBJECT)
                            {
                                Console.WriteLine("このページの" + i
                                                    + " 番目のエレメントがForm XObjectでした。内容を確認します。");
                                using (PtlEditFormXObject formXObject = (PtlEditFormXObject)element)
                                using (PtlContent contentInXObject = formXObject.getContent())// FormXObjectに含まれる
                                using (PtlEditElements elementsInXObject = contentInXObject.getEditElements())
                                {
                                    getFormXObjectInfo(elementsInXObject);
                                }
                            }
                        }
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void getFormXObjectInfo(PtlEditElements elems)
        {
            try
            {
                int numOfElements = elems.getCount();
                Console.WriteLine("このFormXObjectに含まれるエレメントの数は" + numOfElements + "です。");


                for (int i = 0; i < numOfElements; i++)
                {

                    Console.WriteLine("---このFormXObjectの" + i + "番目のエレメントの内容を確認します。---");
                    using (PtlEditElement element = elems.get(i))
                    {
                        //エレメントに共通の情報（BBox, ClippedBox）の情報を表示
                        getElementInfo(element);

                        switch (element.getType())
                        {
                            case PtlEditElement.ELEMENT_TYPE.TYPE_FORMXOBJECT:
                                Console.WriteLine("このエレメントは入れ子になったForm XObjectでした。内容を確認します。");
                                using (PtlEditFormXObject formXObject = (PtlEditFormXObject)element)
                                using (PtlContent contentInXObject = formXObject.getContent())// FormXObjectに含まれるContentを取得
                                using (PtlEditElements elementsInXObject = contentInXObject.getEditElements())
                                {
                                    Console.WriteLine("--");
                                    getFormXObjectInfo(elementsInXObject);
                                }
                                break;


                            case PtlEditElement.ELEMENT_TYPE.TYPE_TEXT:
                                getEditTextInfo(element);
                                break;

                            case PtlEditElement.ELEMENT_TYPE.TYPE_PATH:
                                getEditPathInfo(element);
                                break;

                            case PtlEditElement.ELEMENT_TYPE.TYPE_IMAGE:
                                getEditImageInfo(element);
                                break;
                        }
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void getElementInfo(PtlEditElement element)
        {
            try
            {
                Console.WriteLine("エレメントのBBoxの情報を表示します。");
                //BBoxの情報を印字
                using (PtlRect bBox = element.getBBox())
                {
                    getRectInfo(bBox, "このエレメントのBBox");
                }

                //ClippedBoxの情報を印字
                if (element.hasClip())
                {
                    using (PtlRect clippedBox = element.getClippedBBox())
                    {
                        Console.WriteLine("このエレメントにはクリッピングがあります。");
                        getRectInfo(clippedBox, "このエレメントのClippedBox");
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void getEditPathInfo(PtlEditElement element)
        {
            try
            {
                using (PtlEditPath pathElement = (PtlEditPath)element)
                using (PtlEditPathItems pathItems = pathElement.getPathItems())
                {
                    Console.WriteLine("PATHエレメント固有の情報を表示します。");
                    Console.WriteLine("このPATHエレメントのペイントフラグ情報を表示します。");
                    Console.WriteLine("-");
                    Console.Write("ペイントフラグ：");
                    PtlEditPath.PAINT_FLAGS paintFlag = pathElement.getPaintFlags();
                    switch (paintFlag)
                    {
                        case PtlEditPath.PAINT_FLAGS.PAINT_EOFILL: //パスを塗りつぶします。奇偶規則。
                            Console.WriteLine("奇偶規則に基づいてパスを塗りつぶす");
                            break;

                        case PtlEditPath.PAINT_FLAGS.PAINT_FILL: //パスを塗りつぶします。非ゼロ回転規則。
                            Console.WriteLine("非ゼロ回転規則に基づいてパスを塗りつぶす");
                            break;

                        case PtlEditPath.PAINT_FLAGS.PAINT_INVISIBLE: //ストロークも塗りつぶしも行いません。
                            Console.WriteLine("ストロークも塗りつぶしも行わない");
                            break;

                        case PtlEditPath.PAINT_FLAGS.PAINT_STROKE: //ストロークします。
                            Console.WriteLine("ストロークする");
                            break;

                        default:
                            Console.WriteLine("不明なフラグが取得されました。");
                            Console.WriteLine("paintFlag = " + paintFlag);
                            break;
                    }
                    //ストロークカラー情報の取得
                    getStrokeColorInfo(pathElement);
                    //塗りつぶしカラー情報の取得
                    getFillColorInfo(pathElement);
                    //パスアイテムの情報取得
                    getPathItemsInfo(pathItems);
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void getPathItemsInfo(PtlEditPathItems pathItems)
        {
            try
            {
                Console.WriteLine("-");
                Console.WriteLine("パスアイテムの情報を表示します。");
                for (int j = 0; j < pathItems.getCount(); j++)
                {
                    using (PtlEditPathItem item = pathItems.get(j))
                    {
                        PtlEditPathItem.PATH_ITEM_TYPE itemType = item.getType();

                        Console.WriteLine(j + "番目のアイテム情報：");
                        Console.WriteLine("---");
                        switch (itemType)
                        {
                            case PtlEditPathItem.PATH_ITEM_TYPE.TYPE_PATH_CURVE:
                                Console.WriteLine("アイテム種別：ベジェ曲線");
                                using (PtlEditPathCurve pathCurve = (PtlEditPathCurve)item)
                                {
                                    getPathCurveInfo(pathCurve);
                                }
                                break;
                            case PtlEditPathItem.PATH_ITEM_TYPE.TYPE_PATH_LINE:
                                Console.WriteLine("アイテム種別：直線");
                                using (PtlEditPathLine pathLine = (PtlEditPathLine)item)
                                {
                                    getPathLineInfo(pathLine);
                                }
                                break;
                            case PtlEditPathItem.PATH_ITEM_TYPE.TYPE_PATH_RECT:
                                Console.WriteLine("アイテム種別：矩形");
                                using (PtlEditPathRect pathRect = (PtlEditPathRect)item)
                                using (PtlRect rectOfPath = pathRect.getRect())
                                {
                                    getRectInfo(rectOfPath, "このパスアイテムの矩形");
                                }
                                break;
                        }
                        Console.WriteLine("---");
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void getPathCurveInfo(PtlEditPathCurve pathCurve)
        {
            try
            {
                using (PtlPoint startPoint = pathCurve.getStartPoint())
                using (PtlPoint endPoint = pathCurve.getEndPoint())
                using (PtlPoint controlPoint1 = pathCurve.getControlPoint1())
                using (PtlPoint controlPoint2 = pathCurve.getControlPoint2())
                {
                    Console.Write("開始位置[x, y]: ");
                    Console.WriteLine("[" + startPoint.getX() + ", " + startPoint.getY() + "]");
                    Console.Write("終了位置[x, y]: ");
                    Console.WriteLine("[" + endPoint.getX() + ", " + endPoint.getY() + "]");
                    Console.Write("第一の制御点[x, y]: ");
                    Console.WriteLine("[" + controlPoint1.getX() + ", " + controlPoint1.getY() + "]");
                    Console.Write("第二の制御点[x, y]: ");
                    Console.WriteLine("[" + controlPoint2.getX() + ", " + controlPoint2.getY() + "]");
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void getPathLineInfo(PtlEditPathLine pathLine)
        {
            try
            {
                using (PtlPoint startPoint = pathLine.getStartPoint())
                using (PtlPoint endPoint = pathLine.getEndPoint())
                {
                    Console.Write("開始位置[x, y]: ");
                    Console.WriteLine("[" + startPoint.getX() + ", " + startPoint.getY() + "]");
                    Console.Write("終了位置[x, y]: ");
                    Console.WriteLine("[" + endPoint.getX() + ", " + endPoint.getY() + "]");
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * 読み込まれたページに含まれるTEXTエレメントが持つ情報を表示する関数です。
         * @param page 読み込まれたページ
         * @throws PtlException
         * @throws Exception
         */
        private static void getEditTextInfo(PtlEditElement element)
        {
            try
            {
                using (PtlEditText textElement = (PtlEditText)element)
                using (PtlEditTextItems textItems = textElement.getTextItems())
                {
                    Console.WriteLine("TEXTエレメント固有の情報を表示します。");
                    Console.WriteLine("テキストアイテムの情報を表示します。");
                    getTextItemsInfo(textItems);
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * Textアイテムが持つ情報を表示する関数です
         * @param textItems そのエレメントが持つTextItemsコンテナクラス
         * @throws PtlException
         * @throws Exception
         */
        private static void getTextItemsInfo(PtlEditTextItems textItems)
        {
            try
            {
                int numOfTextItems = textItems.getCount();
                Console.WriteLine("このテキストアイテムに含まれるアイテム数は" + numOfTextItems + "です。");
                for (int j = 0; j < numOfTextItems; j++)
                {
                    Console.WriteLine(j + "番目のアイテム情報：");
                    Console.WriteLine("---");

                    using (PtlEditTextItem item = textItems.get(j))
                    using (PtlRect textItemBBox = item.getBBox())
                    using (PtlFontInfo fontInfo = item.getFontInfo())
                    {
                        Console.WriteLine("テキストアイテムのBBox情報：");
                        getRectInfo(textItemBBox, "このテキストアイテムのBBox");

                        Console.WriteLine("-");
                        Console.WriteLine("テキストアイテムが含むテキスト：");
                        Console.WriteLine(item.getText());

                        Console.WriteLine("-");
                        Console.WriteLine("フォント情報：");
                        Console.WriteLine("フォント名：" + fontInfo.getFontName());
                        Console.WriteLine("フォントタイプ：" + fontInfo.getFontType());
                        Console.WriteLine("エンコーディングタイプ：" + fontInfo.getEncodingType());
                        if (fontInfo.getEncodingType() == PtlFontInfo.ENCODING_TYPE.ENCODING_NAME)
                        {
                            Console.WriteLine("エンコーディング名：" + fontInfo.getEncodingName());
                        }
                        if (fontInfo.isEmbedded())
                        {
                            Console.WriteLine("フォント埋め込み：あり");
                        }
                        else
                        {
                            Console.WriteLine("フォント埋め込み：なし");
                        }

                        Console.WriteLine("-");
                        Console.Write("ペイントフラグ：");
                        PtlEditPath.PAINT_FLAGS paintFlag = (PtlEditPath.PAINT_FLAGS)item.getPaintFlags();
                        switch (paintFlag)
                        {
                            case PtlEditPath.PAINT_FLAGS.PAINT_FILL: //パスを塗りつぶす
                                Console.WriteLine("パスを塗りつぶす");
                                Console.WriteLine("塗りつぶしカラー情報を取得します。");
                                getFillColorInfo(item); //塗りつぶしカラー情報を取得するための関数
                                break;

                            case PtlEditPath.PAINT_FLAGS.PAINT_INVISIBLE: //ストロークも塗りつぶしも行いません。
                                Console.WriteLine("ストロークも塗りつぶしも行わない");
                                break;

                            case PtlEditPath.PAINT_FLAGS.PAINT_STROKE: //ストロークする
                                Console.WriteLine("ストロークする");
                                Console.WriteLine("ストロークカラー情報を取得します");
                                getStrokeColorInfo(item); //ストロークカラー情報を取得するための関数
                                break;

                            default:
                                Console.WriteLine("不明なフラグが取得されました。");
                                Console.WriteLine("paintFlag = " + paintFlag);
                                break;
                        }
                        Console.WriteLine("---");
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        public static void getEditImageInfo(PtlEditElement element)
        {
            try
            {
                // ページコンテント・画像エレメントの取得
                // OUTPUT_FORMATはFORMAT_AUTOで設定。
                // 画像エレメントの取得
                using (PtlEditImage elemImage = (PtlEditImage)element)
                {
                    Console.WriteLine("画像の高さ：" + elemImage.getHeight());
                    Console.WriteLine("画像の幅　：" + elemImage.getWidth());
                    Console.WriteLine("画像のPPI ：" + elemImage.getPPI());
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * ストロークカラー情報を取得するための関数。
         * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
         * @param pathElement パスエレメントを表現したクラス
         * @throws PtlException
         * @throws Exception
         */
        private static void getStrokeColorInfo(PtlEditPath pathElement)
        {
            try
            {
                Console.WriteLine("-");
                Console.WriteLine("ストロークカラー情報：");
                using (PtlColorSpace strokeColorSpace = pathElement.getStrokeColorSpace())
                using (PtlColor strokeColor = pathElement.getStrokeColor())
                {

                    PtlColorSpace.COLOR_SPACE_TYPE colorSpaceType = strokeColorSpace.getType();
                    switch (colorSpaceType)
                    {
                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICECMYK:
                            using (PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)strokeColor)
                            {
                                getCMYKInfo(deviceCMYK);
                            }
                            break;


                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICEGRAY:
                            using (PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)strokeColor)
                            {
                                getGrayInfo(deviceGray);
                            }
                            break;


                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICERGB:
                            using (PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)strokeColor)
                            {
                                getRGBInfo(deviceRGB);
                            }
                            break;

                        default:
                            Console.WriteLine("ストロークカラーは"
                                + "読み取れるカラースペースタイプではありませんでした。");
                            Console.WriteLine("カラースペースタイプ : " + colorSpaceType);
                            break;
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
        /**
         * ストロークカラー情報を取得するための関数。
         * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
         * @param item PtlEditTextItem テキストアイテムを表現したクラス
         * @throws PtlException 
         * @throws Exception 
         */
        private static void getStrokeColorInfo(PtlEditTextItem item)
        {
            try
            {
                Console.WriteLine("-");
                Console.WriteLine("ストロークカラー情報：");

                using (PtlColor strokeColor = item.getStrokeColor())
                {
                    switch (strokeColor.getType())
                    {
                        case PtlColor.COLOR_TYPE.TYPE_DEVICE_CMYK:
                            using (PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)item.getStrokeColor())
                            {
                                getCMYKInfo(deviceCMYK);
                            }
                            break;

                        case PtlColor.COLOR_TYPE.TYPE_DEVICE_GRAY:
                            using (PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)item.getStrokeColor())
                            {
                                getGrayInfo(deviceGray);
                            }
                            break;

                        case PtlColor.COLOR_TYPE.TYPE_DEVICE_RGB:
                            using (PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)item.getStrokeColor())
                            {
                                getRGBInfo(deviceRGB);
                            }
                            break;

                        default:
                            Console.WriteLine("ストロークカラーの色の値は取得できませんでした。");
                            Console.WriteLine("色の値 : " + strokeColor.getType());
                            break;
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * 塗りつぶしカラー情報を取得するための関数。
         * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
         * @param pathElement パスエレメントを表現したクラス
         * @throws PtlException 
         * @throws Exception 
         */
        private static void getFillColorInfo(PtlEditPath pathElement)
        {
            try
            {
                Console.WriteLine("-");
                Console.WriteLine("塗りつぶしカラー情報：");
                using (PtlColorSpace fillColorSpace = pathElement.getStrokeColorSpace())
                using (PtlColor fillColor = pathElement.getStrokeColor())
                {

                    PtlColorSpace.COLOR_SPACE_TYPE colorSpaceType = fillColorSpace.getType();
                    switch (colorSpaceType)
                    {
                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICECMYK:
                            using (PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)fillColor)
                            {
                                getCMYKInfo(deviceCMYK);
                            }
                            break;


                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICEGRAY:
                            using (PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)fillColor)
                            {
                                getGrayInfo(deviceGray);
                            }
                            break;

                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICERGB:
                            using (PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)fillColor)
                            {
                                getRGBInfo(deviceRGB);
                            }
                            break;

                        default:
                            Console.WriteLine("ストロークカラーは"
                                + "読み取れるカラースペースタイプではありませんでした。");
                            Console.WriteLine("カラースペースタイプ : " + colorSpaceType);
                            break;
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * 塗りつぶしカラー情報を取得するための関数。
         * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
         * @param item PtlEditTextItem テキストアイテムを表現したクラス
         * @throws PtlException 
         * @throws Exception 
         */
        private static void getFillColorInfo(PtlEditTextItem item)
        {
            try
            {
                Console.WriteLine("-");
                Console.WriteLine("塗りつぶしカラー情報：");

                using (PtlColor fillColor = item.getFillColor())
                {
                    switch (fillColor.getType())
                    {
                        case PtlColor.COLOR_TYPE.TYPE_DEVICE_CMYK:
                            using (PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)item.getFillColor())
                            {
                                getCMYKInfo(deviceCMYK);
                            }
                            break;

                        case PtlColor.COLOR_TYPE.TYPE_DEVICE_GRAY:
                            using (PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)item.getFillColor())
                            {
                                getGrayInfo(deviceGray);
                            }
                            break;

                        case PtlColor.COLOR_TYPE.TYPE_DEVICE_RGB:
                            using (PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)item.getFillColor())
                            {
                                getRGBInfo(deviceRGB);
                            }
                            break;

                        default:
                            Console.WriteLine("塗りつぶしカラーの色の値は取得できませんでした。");
                            Console.WriteLine("色の値 : " + fillColor.getType());
                            break;
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * deviceCYMKの各要素の数値を表示する関数です
         * @param cmyk
         * @throws PtlException
         * @throws Exception
         */
        private static void getCMYKInfo(PtlColorDeviceCMYK cmyk)
        {
            try
            {
                float paramC = cmyk.getC();
                float paramM = cmyk.getM();
                float paramY = cmyk.getY();
                float paramK = cmyk.getK();

                Console.WriteLine("DeviceCMYK色の要素[C, M, Y, K]の値は、");
                Console.WriteLine("[" + paramC + ", " + paramM + ", " +
                                         paramY + ", " + paramK + "]です。");
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * deviceGrayの各要素の数値を表示する関数です
         * @param gray
         * @throws PtlException
         * @throws Exception
         */
        private static void getGrayInfo(PtlColorDeviceGray gray)
        {
            try
            {
                float paramGray = gray.getGray();

                Console.WriteLine("DeviceGray色のGray要素の値は、");
                Console.WriteLine("[" + paramGray + "]です。");
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * deviceRGBの各要素の数値を表示する関数です
         * @param rgb
         * @throws PtlException
         * @throws Exception
         */
        private static void getRGBInfo(PtlColorDeviceRGB rgb)
        {
            try
            {
                float paramR = rgb.getR();
                float paramG = rgb.getG();
                float paramB = rgb.getB();

                Console.WriteLine("DeviceRGB色の[R, G, B]の要素の値は、");
                Console.WriteLine("[" + paramR + ", " + paramG + ", " +
                                         paramB + "]です。");
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * PtlRectの各情報を表示する関数です。
         * @param rect
         * @param rectName
         * @throws PtlException
         * @throws Exception
         */
        private static void getRectInfo(PtlRect rect, String rectName)
        {
            try
            {
                float top = rect.getTop();
                float bottom = rect.getBottom();
                float left = rect.getLeft();
                float right = rect.getRight();
                Console.WriteLine(rectName + "の[top, bottom, left, right]は、");
                Console.WriteLine("[" + top + ", " + bottom + ", " + left + ", " + right + "]です。");
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
    }
}
