﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：Contentに含まれるPATHエレメントの情報を表示します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace GetPtlEditPathInfo
{
    internal class GetPtlEditPathInfo
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: GetPtlEditPathInfo in-pdf-file");
        }

        static void Main(string[] args)
        {
            if (args.Length < 1)
            {
                printUsage();
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    //ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                      // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            TextWriter errorWriter = Console.Error;
                            errorWriter.WriteLine("入力PDFのページコンテナが空");
                            return;
                        }
                        // ページ数の取得
                        for (int i = 0; i < pages.getCount(); i++)
                        {
                            Console.WriteLine("-----------------");
                            Console.WriteLine(i + "ページ目の情報を表示します。");
                            using (PtlPage page = pages.get(i))
                            {
                                getPathInfo(page);
                            }
                            Console.WriteLine("-----------------");
                        }
                    }
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        private static void getPathInfo(PtlPage page)
        {
            try
            {
                using (PtlContent content = page.getContent())// 挿入先ページコンテントの取得
                using (PtlEditElements elems = content.getEditElements(PtlContent.GET_ELEMENT.GET_PATH))
                {
                    int numOfElements = elems.getCount();
                    Console.WriteLine("このページに含まれるPATHエレメントの数は" + numOfElements + "です。");

                    for (int i = 0; i < numOfElements; i++)
                    {
                        Console.WriteLine("---" + i + "番目のPATHエレメントの情報を表示します。---");
                        using (PtlEditPath pathElement = (PtlEditPath)elems.get(i))
                        using (PtlEditPathItems pathItems = pathElement.getPathItems())
                        {
                            //エレメントに共通の情報（BBox, ClippedBox）の情報を表示
                            getElementInfo(pathElement);

                            Console.WriteLine("PATHエレメント固有の情報を表示します。");
                            Console.WriteLine("このPATHエレメントのペイントフラグ情報を表示します。");
                            Console.WriteLine("-");
                            Console.Write("ペイントフラグ：");
                            PtlEditPath.PAINT_FLAGS paintFlag = pathElement.getPaintFlags();
                            switch (paintFlag)
                            {
                                case PtlEditPath.PAINT_FLAGS.PAINT_EOFILL: //パスを塗りつぶします。奇偶規則。
                                    Console.WriteLine("奇偶規則に基づいてパスを塗りつぶす");
                                    break;

                                case PtlEditPath.PAINT_FLAGS.PAINT_FILL: //パスを塗りつぶします。非ゼロ回転規則。
                                    Console.WriteLine("非ゼロ回転規則に基づいてパスを塗りつぶす");
                                    break;

                                case PtlEditPath.PAINT_FLAGS.PAINT_INVISIBLE: //ストロークも塗りつぶしも行いません。
                                    Console.WriteLine("ストロークも塗りつぶしも行わない");
                                    break;

                                case PtlEditPath.PAINT_FLAGS.PAINT_STROKE: //ストロークします。
                                    Console.WriteLine("ストロークする");
                                    break;

                                default:
                                    Console.WriteLine("不明なフラグが取得されました。");
                                    Console.WriteLine("paintFlag = " + paintFlag);
                                    break;
                            }

                            //ストロークカラー情報の取得
                            getStrokeColorInfo(pathElement);

                            //塗りつぶしカラー情報の取得
                            getFillColorInfo(pathElement);

                            //パスアイテムの情報取得
                            getPathItemsInfo(pathItems);
                        }
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * エレメントが共通して持つ情報を表示する関数です
         * @param element
         * @throws PtlException
         * @throws Exception
         * @throws Error 
         */
        private static void getElementInfo(PtlEditElement element)
        {
            Console.WriteLine("エレメントのBBoxの情報を表示します。");
            //BBoxの情報を印字
            try
            {
                using (PtlRect bBox = element.getBBox())
                {
                    getRectInfo(bBox, "このエレメントのBBox");
                }

                //ClippedBoxの情報を印字
                if (element.hasClip())
                {
                    using (PtlRect clippedBox = element.getClippedBBox())
                    {
                        Console.WriteLine("このエレメントにはクリッピングがあります。");
                        getRectInfo(clippedBox, "このエレメントのClippedBox");
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * ストロークカラー情報を取得するための関数。
         * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
         * @param pathElement パスエレメントを表現したクラス
         * @throws PtlException
         * @throws Exception
         */
        private static void getStrokeColorInfo(PtlEditPath pathElement)
        {
            try
            {
                Console.WriteLine("-");
                Console.WriteLine("ストロークカラー情報：");
                using (PtlColorSpace strokeColorSpace = pathElement.getStrokeColorSpace())
                using (PtlColor strokeColor = pathElement.getStrokeColor())
                {
                    PtlColorSpace.COLOR_SPACE_TYPE colorSpaceType = strokeColorSpace.getType();
                    switch (colorSpaceType)
                    {
                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICECMYK:
                            using (PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)strokeColor)
                            {
                                getCMYKInfo(deviceCMYK);
                            }
                            break;
                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICEGRAY:
                            using (PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)strokeColor)
                            {
                                getGrayInfo(deviceGray);
                            }
                            break;
                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICERGB:
                            using (PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)strokeColor)
                            {
                                getRGBInfo(deviceRGB);
                            }
                            break;
                        default:
                            Console.WriteLine("ストロークカラーは"
                                + "読み取れるカラースペースタイプではありませんでした。");
                            Console.WriteLine("カラースペースタイプ : " + colorSpaceType);
                            break;
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * 塗りつぶしカラー情報を取得するための関数。
         * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
         * @param pathElement パスエレメントを表現したクラス
         * @throws PtlException 
         * @throws Exception 
         */
        private static void getFillColorInfo(PtlEditPath pathElement)
        {
            try
            {
                Console.WriteLine("-");
                Console.WriteLine("塗りつぶしカラー情報：");
                using (PtlColorSpace fillColorSpace = pathElement.getStrokeColorSpace())
                using (PtlColor fillColor = pathElement.getStrokeColor())
                {
                    PtlColorSpace.COLOR_SPACE_TYPE colorSpaceType = fillColorSpace.getType();
                    switch (colorSpaceType)
                    {
                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICECMYK:
                            using (PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)fillColor)
                            {
                                getCMYKInfo(deviceCMYK);
                            }
                            break;
                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICEGRAY:
                            using (PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)fillColor)
                            {
                                getGrayInfo(deviceGray);
                            }
                            break;
                        case PtlColorSpace.COLOR_SPACE_TYPE.TYPE_DEVICERGB:
                            using (PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)fillColor)
                            {
                                getRGBInfo(deviceRGB);
                            }
                            break;
                        default:
                            Console.WriteLine("ストロークカラーは"
                                + "読み取れるカラースペースタイプではありませんでした。");
                            Console.WriteLine("カラースペースタイプ : " + colorSpaceType);
                            break;
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * deviceCYMKの各要素の数値を表示する関数です
         * @param cmyk
         * @throws PtlException
         * @throws Exception
         */
        private static void getCMYKInfo(PtlColorDeviceCMYK cmyk)
        {
            try
            {
                float paramC = cmyk.getC();
                float paramM = cmyk.getM();
                float paramY = cmyk.getY();
                float paramK = cmyk.getK();

                Console.WriteLine("DeviceCMYK色の要素[C, M, Y, K]の値は、");
                Console.WriteLine("[" + paramC + ", " + paramM + ", " + paramY + ", " + paramK + "]です。");
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * deviceGrayの各要素の数値を表示する関数です
         * @param gray
         * @throws PtlException
         * @throws Exception
         */
        private static void getGrayInfo(PtlColorDeviceGray gray)
        {
            try
            {
                float paramGray = gray.getGray();

                Console.WriteLine("DeviceGray色のGray要素の値は、");
                Console.WriteLine("[" + paramGray + "]です。");
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * deviceRGBの各要素の数値を表示する関数です
         * @param rgb
         * @throws PtlException
         * @throws Exception
         */
        private static void getRGBInfo(PtlColorDeviceRGB rgb)
        {
            try
            {
                float paramR = rgb.getR();
                float paramG = rgb.getG();
                float paramB = rgb.getB();

                Console.WriteLine("DeviceRGB色の[R, G, B]の要素の値は、");
                Console.WriteLine("[" + paramR + ", " + paramG + ", " + paramB + "]です。");
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void getPathItemsInfo(PtlEditPathItems pathItems)
        {
            try
            {
                Console.WriteLine("-");
                Console.WriteLine("パスアイテムの情報を表示します。");
                for (int j = 0; j < pathItems.getCount(); j++)
                {
                    using (PtlEditPathItem item = pathItems.get(j))
                    {
                        PtlEditPathItem.PATH_ITEM_TYPE itemType = item.getType();

                        Console.WriteLine(j + "番目のアイテム情報：");
                        Console.WriteLine("---");
                        switch (itemType)
                        {
                            case PtlEditPathItem.PATH_ITEM_TYPE.TYPE_PATH_CURVE:
                                Console.WriteLine("アイテム種別：ベジェ曲線");
                                using (PtlEditPathCurve pathCurve = (PtlEditPathCurve)item)
                                {
                                    getPathCurveInfo(pathCurve);
                                }
                                break;

                            case PtlEditPathItem.PATH_ITEM_TYPE.TYPE_PATH_LINE:
                                Console.WriteLine("アイテム種別：直線");
                                using (PtlEditPathLine pathLine = (PtlEditPathLine)item)
                                {
                                    getPathLineInfo(pathLine);
                                }
                                break;

                            case PtlEditPathItem.PATH_ITEM_TYPE.TYPE_PATH_RECT:
                                Console.WriteLine("アイテム種別：矩形");
                                using (PtlEditPathRect pathRect = (PtlEditPathRect)item)
                                using (PtlRect rectOfPath = pathRect.getRect())
                                {
                                    getRectInfo(rectOfPath, "このパスアイテムの矩形");
                                }
                                break;
                        }
                        Console.WriteLine("---");
                    }
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void getPathCurveInfo(PtlEditPathCurve pathCurve)
        {
            try
            {
                using (PtlPoint startPoint = pathCurve.getStartPoint())
                using (PtlPoint endPoint = pathCurve.getEndPoint())
                using (PtlPoint controlPoint1 = pathCurve.getControlPoint1())
                using (PtlPoint controlPoint2 = pathCurve.getControlPoint2())
                {
                    Console.Write("開始位置[x, y]: ");
                    Console.WriteLine("[" + startPoint.getX() + ", " + startPoint.getY() + "]");
                    Console.Write("終了位置[x, y]: ");
                    Console.WriteLine("[" + endPoint.getX() + ", " + endPoint.getY() + "]");
                    Console.Write("第一の制御点[x, y]: ");
                    Console.WriteLine("[" + controlPoint1.getX() + ", " + controlPoint1.getY() + "]");
                    Console.Write("第二の制御点[x, y]: ");
                    Console.WriteLine("[" + controlPoint2.getX() + ", " + controlPoint2.getY() + "]");
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private static void getPathLineInfo(PtlEditPathLine pathLine)
        {
            try
            {
                using (PtlPoint startPoint = pathLine.getStartPoint())
                using (PtlPoint endPoint = pathLine.getEndPoint())
                {
                    Console.Write("開始位置[x, y]: ");
                    Console.WriteLine("[" + startPoint.getX() + ", " + startPoint.getY() + "]");
                    Console.Write("終了位置[x, y]: ");
                    Console.WriteLine("[" + endPoint.getX() + ", " + endPoint.getY() + "]");
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        /**
         * PtlRectの各情報を表示する関数です。
         * @param rect
         * @param rectName
         * @throws PtlException
         * @throws Exception
         */
        private static void getRectInfo(PtlRect rect, String rectName)
        //throws PtlException, Exception, Error 
        {
            try
            {
                float top = rect.getTop();
                float bottom = rect.getBottom();
                float left = rect.getLeft();
                float right = rect.getRight();
                Console.WriteLine(rectName + "の[top, bottom, left, right]は、");
                Console.WriteLine("[" + top + ", " + bottom + ", " + left + ", " + right + "]です。");
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
    }
}
