﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：テキストボックスによる描画の際にテキストの色などを指定する

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace TBoxDrawTextBoxSetTextParams
{
    internal class TBoxDrawTextBoxSetTextParams
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: TBoxDrawTextBoxSetTextParams"
                    + " in-pdf-file out-pdf-file"
                    + " page-num text-to-add text-align"
                    + " font-family font-size"
                    + " text-color-R text-color-G text-color-B"
                    + " text-outline-color-R text-outline-color-G text-outline-color-B"
                    + " text-opacity");
        }

        static void Main(string[] args)
        {
            if (args.Length < 13)
            {
                printUsage();
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // コマンドライン引数の読み取り・判定
                    int pageToAdd = int.Parse(args[2]);
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);
                    if ((numPages < 0) || (numPages < pageToAdd))
                    {
                        TextWriter errorWriter = Console.Error;
                        errorWriter.WriteLine("テキスト挿入ページは全ページ数よりも小さい正の値を指定してください。");
                        return;
                    }

                    String textToAdd = args[3];
                    String textAlign = args[4];
                    String fontFamily = args[5];
                    float fontSize = float.Parse(args[6]);
                    float textR = float.Parse(args[7]);
                    float textG = float.Parse(args[8]);
                    float textB = float.Parse(args[9]);
                    float textOutlineR = float.Parse(args[10]);
                    float textOutlineG = float.Parse(args[11]);
                    float textOutlineB = float.Parse(args[12]);
                    float opacity = float.Parse(args[13]);
                    if ((opacity < 0) || (1.0f < opacity))
                    {
                        TextWriter errorWriter = Console.Error;
                        errorWriter.WriteLine("0.0から1.0の範囲で値を設定し直してください。");
                        return;
                    }

                    //ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            TextWriter errorWriter = Console.Error;
                            errorWriter.WriteLine("ページコンテナが空");
                            return;
                        }
                        try
                        {
                            using (PtlPage page = pages.get(pageToAdd - 1)) // ページの取得(パラメータindexは0が先頭のため1を引く)
                            using (PtlContent content = page.getContent())  // ページコンテントの取得
                            using (PtlRect outputRect = new PtlRect())      // 出力矩形の初期化
                            using (PtlTextBox textBox = content.drawTextBox(setRectCoordinate(outputRect), valueAlign(textAlign), 210, 297))
                            using (PtlParamWriteStringTextBox textParam = new PtlParamWriteStringTextBox()) // 文字描画のパラメータクラス。
                            using (PtlParamFont font = new PtlParamFont())
                            using (PtlColorDeviceRGB textColor = new PtlColorDeviceRGB(textR, textG, textB))
                            using (PtlColorDeviceRGB outlineColor = new PtlColorDeviceRGB(textOutlineR, textOutlineG, textOutlineB))
                            {
                                // textParamの指定
                                // フォントの各種項目を指定
                                font.setName(fontFamily);
                                font.setSize(fontSize);
                                // パラメータにフォントを設定
                                textParam.setFont(font);
                                // 文字の色指定
                                textParam.setTextColor(textColor);
                                // 文字の背景色の指定
                                textParam.setOutlineColor(outlineColor);
                                // 文字の透明度の指定
                                textParam.setOpacity(opacity);

                                // 文字列出力
                                textBox.writeString(textToAdd, textParam);
                                textBox.terminate();
                            }
                        }
                        catch (ArgumentException ex)
                        {
                            //PtlContent.ALIGN指定のエラー処理
                            Console.WriteLine(ex.Message);
                            Console.WriteLine("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                            Console.WriteLine(ex.StackTrace);
                        }
                    }
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (FormatException numfe)
            {
                Console.WriteLine("RGB及び透明度の指定には数値を使用して下さい。");
                Console.WriteLine(numfe.Message);
                Console.WriteLine(numfe.StackTrace);
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() + "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        /**
         * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         * 特に外部からの呼び出しを想定しないためprivateとする。
         * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         * @throws IOException
         * @throws PtlException
         * @throws Exception
         */
        public static PtlRect setRectCoordinate(PtlRect outputRect)
        {
            try
            {
                float top, bottom, left, right;
                bool isValueOkay = false;

                while (!isValueOkay)
                {
                    Console.WriteLine("指定する矩形の各数値を入力してください。");
                    Console.Write("top (mm) : ");
                    top = float.Parse(Console.ReadLine());
                    Console.Write("bottom (mm) : ");
                    bottom = float.Parse(Console.ReadLine());
                    //不正矩形は再入力させる
                    if (top < bottom)
                    {
                        Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                        continue;
                    }
                    Console.Write("left (mm) : ");
                    left = float.Parse(Console.ReadLine());
                    Console.Write("right (mm) : ");
                    right = float.Parse(Console.ReadLine());
                    //不正矩形は再入力させる
                    if (right < left)
                    {
                        Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                        continue;
                    }

                    //矩形を正しく指定できた場合の処理
                    isValueOkay = true;
                    outputRect.setLeft(left);
                    outputRect.setBottom(bottom);
                    outputRect.setRight(right);
                    outputRect.setTop(top);
                }
                return outputRect;
            }
            catch (IOException ex)
            {
                throw ex;
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        // 文字列を PtlContent.ALIGN として返す
        static PtlContent.ALIGN valueAlign(String textAlign)
        {
            if (textAlign.Equals(PtlContent.ALIGN.ALIGN_BOTTOM.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_BOTTOM;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_BOTTOM_LEFT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_BOTTOM_LEFT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_BOTTOM_RIGHT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_BOTTOM_RIGHT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_CENTER.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_CENTER;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_LEFT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_LEFT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_RIGHT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_RIGHT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_TOP.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_TOP;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_TOP_LEFT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_TOP_LEFT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_TOP_RIGHT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_TOP_RIGHT;
            }
            else
            {
                throw new ArgumentException();
            }
        }
    }
}
