﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：テキストボックスを用いて指定したページの指定した位置に
        テキストを挿入します。
  　　　配置矩形のみ、プログラム内で指定します。
  　　　上付き文字・下付き文字・化学式を利用した定型文を挿入します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace TBoxDrawWithFormat
{
    internal class TBoxDrawWithFormat
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: TBoxDrawWithFormat in-pdf-file out-pdf-file page-num text-to-add text-align formatNum");
            Console.WriteLine("FormatNum\n0: ノーマル 1: 化学式 2: 下付き 3: 上付き");
        }

        static void Main(string[] args)
        {
            if (args.Length < 3)
            {
                printUsage();
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // コマンドライン引数の読み取り・判定
                    int pageToAdd = int.Parse(args[2]);
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);
                    if ((numPages < 0) || (numPages < pageToAdd))
                    {
                        TextWriter errorWriter = Console.Error;
                        errorWriter.WriteLine("テキスト挿入ページは全ページ数よりも小さい正の値を指定してください。");
                        return;
                    }
                    String textToAdd = args[3];
                    String textAlign = args[4];
                    //ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            TextWriter errorWriter = Console.Error;
                            errorWriter.WriteLine("ページコンテナが空");
                            return;
                        }

                        try
                        {
                            using (PtlPage page = pages.get(pageToAdd - 1))// ページの取得(パラメータindexは0が先頭のため1を引く)
                            using (PtlContent content = page.getContent())// ページコンテントの取得
                            using (PtlRect outputRect = new PtlRect())// 出力矩形の初期化
                            using (PtlTextBox textBox = content.drawTextBox(setRectCoordinate(outputRect), PtlContent.ALIGN.ALIGN_CENTER, 210, 297))
                            using (PtlParamWriteStringTextBox paramFormat = new PtlParamWriteStringTextBox())
                            using (PtlParamWriteStringTextBox paramFormat_sup = new PtlParamWriteStringTextBox())
                            using (PtlParamWriteStringTextBox paramFormat_sub = new PtlParamWriteStringTextBox())
                            using (PtlParamWriteStringTextBox paramFormat_chem = new PtlParamWriteStringTextBox())
                            {// 文字描画のパラメータクラス。

                                paramFormat.setFormat(PtlParamWriteStringTextBox.FORMAT_TYPE.FORM_NORMAL);
                                paramFormat_sup.setFormat(PtlParamWriteStringTextBox.FORMAT_TYPE.FORM_SUP);
                                paramFormat_sub.setFormat(PtlParamWriteStringTextBox.FORMAT_TYPE.FORM_SUB);
                                paramFormat_chem.setFormat(PtlParamWriteStringTextBox.FORMAT_TYPE.FORM_CHEM);
                                // 文字列出力（上付き文字）
                                textBox.writeString("これは上付き文字を使った表現です。", paramFormat);
                                textBox.writeNL();
                                textBox.writeString("2", paramFormat);
                                textBox.writeString("10", paramFormat_sup);
                                textBox.writeString(" = 1024", paramFormat);
                                textBox.writeNL();
                                textBox.writeString("Microsoft", paramFormat);
                                textBox.writeString("Ⓡ", paramFormat_sup);

                                // 文字列出力（下付き文字）
                                textBox.writeNL();
                                textBox.writeNL();
                                textBox.writeString("これは下付き文字を使った表現です。", paramFormat);
                                textBox.writeNL();
                                textBox.writeString("A", paramFormat);
                                textBox.writeString("i", paramFormat_sub);
                                textBox.writeString("+B", paramFormat);
                                textBox.writeString("i", paramFormat_sub);
                                textBox.writeString("=C", paramFormat);
                                textBox.writeString("i", paramFormat_sub);

                                // 文字列出力（化学式）
                                textBox.writeNL();
                                textBox.writeNL();
                                textBox.writeString("これは化学式を使った表現です。", paramFormat);
                                textBox.writeNL();
                                textBox.writeString("O2 + 2H2 →2H2O", paramFormat_chem);
                                textBox.writeNL();
                                textBox.writeString("Na+ + Cl- → NaCl", paramFormat_chem);
                                textBox.terminate();
                            }
                        }
                        catch (ArgumentException ex)
                        {//PtlContent.ALIGN指定のエラー処理
                            Console.WriteLine(ex.Message);
                            Console.WriteLine("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                            Console.WriteLine(ex.StackTrace);
                        }
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        /**
         * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         * 特に外部からの呼び出しを想定しないためprivateとする。
         * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         * @throws IOException
         * @throws PtlException
         * @throws Exception
         */
        public static PtlRect setRectCoordinate(PtlRect outputRect)
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            try
            {
                while (!isValueOkay)
                {
                    Console.WriteLine("指定する矩形の各数値を入力してください。");
                    Console.Write("top (mm) : ");
                    top = float.Parse(Console.ReadLine());
                    Console.Write("bottom (mm) : ");
                    bottom = float.Parse(Console.ReadLine());
                    if (top < bottom)
                    {
                        //不正矩形は再入力させる
                        Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                        continue;
                    }
                    Console.Write("left (mm) : ");
                    left = float.Parse(Console.ReadLine());
                    Console.Write("right (mm) : ");
                    right = float.Parse(Console.ReadLine());
                    if (right < left)
                    {
                        //不正矩形は再入力させる
                        Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                        continue;
                    }

                    //矩形を正しく指定できた場合の処理
                    isValueOkay = true;
                    outputRect.setLeft(left);
                    outputRect.setBottom(bottom);
                    outputRect.setRight(right);
                    outputRect.setTop(top);
                }
            }
            catch (IOException ex)
            {
                throw ex;
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
            return outputRect;
        }
    }
}
