﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：テキストボックスを用いて指定したページの指定した位置に
        テキストを挿入します。
  　　　配置矩形のみ、プログラム内で指定します。
  　　　取り消し線・下線の有無を引数で指定します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace TBoxDrawWithLine
{
    internal class TBoxDrawWithLine
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: TBoxDrawWithLineTBoxDrawWithLine in-pdf-file out-pdf-file page-num text-to-add text-with-line text-align"
                + "setStrikeOut setUnderline");
            Console.WriteLine("setStrikeOut\n 0: 取り消し線を引かない 1: 取り消し線を引く");
            Console.WriteLine("setUnderline\n 0: 下線を引かない 1: 下線を引く");
        }

        static void Main(string[] args)
        {
            if (args.Length < 8)
            {
                printUsage();
                return;

            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);

                    // コマンドライン引数の読み取り・判定
                    int pageToAdd = int.Parse(args[2]);
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);
                    if ((numPages < 0) || (numPages < pageToAdd))
                    {
                        TextWriter errorWriter = Console.Error;
                        errorWriter.WriteLine("テキスト挿入ページは全ページ数よりも小さい正の値を指定してください。");
                        return;
                    }

                    String textToAdd = args[3];
                    String textToAddWithLine = args[4];
                    String textAlign = args[5];
                    bool setStrikeOut = true;
                    try
                    {
                        setStrikeOut =
                            readBoolArgs(args[6], "setStrikeOut は 0か1で指定してください。");
                    }
                    catch (ArgumentException ex)
                    {
                        Console.WriteLine(ex.Message);
                        printUsage(); // usageメッセージの表示
                        return;
                    }

                    bool setUnderline = true;
                    try
                    {
                        setUnderline =
                            readBoolArgs(args[7], "setStrikeOut は 0か1で指定してください。");
                    }
                    catch (ArgumentException ex)
                    {
                        Console.WriteLine(ex.Message);
                        printUsage(); // usageメッセージの表示
                        return;
                    }

                    //ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            TextWriter errorWriter = Console.Error;
                            errorWriter.WriteLine("ページコンテナが空");
                            return;
                        }

                        try
                        {
                            using (PtlPage page = pages.get(pageToAdd - 1)) // ページの取得(パラメータindexは0が先頭のため1を引く)
                            using (PtlContent content = page.getContent())  // ページコンテントの取得
                            using (PtlRect outputRect = new PtlRect())      // 出力矩形の初期化
                            using (PtlTextBox textBox = content.drawTextBox(setRectCoordinate(outputRect), valueAlign(textAlign), 210, 297))
                            using (PtlParamWriteStringTextBox paramNormal = new PtlParamWriteStringTextBox())
                            // 文字描画のパラメータクラス。今回は何も設定しない。
                            using (PtlParamWriteStringTextBox paramLine = new PtlParamWriteStringTextBox())
                            {
                                paramLine.setStrikeOut(setStrikeOut);
                                paramLine.setUnderline(setUnderline);

                                // 文字列出力
                                textBox.writeString(textToAdd, paramNormal);
                                textBox.writeString(textToAddWithLine, paramLine);
                                textBox.terminate();
                            }
                        }
                        catch (ArgumentException ex)
                        {
                            //PtlContent.ALIGN指定のエラー処理
                            Console.WriteLine(ex.Message);
                            Console.WriteLine("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                            Console.WriteLine(ex.StackTrace);
                        }
                    }
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        /**
         * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         * 特に外部からの呼び出しを想定しないためprivateとする。
         * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         * @throws IOException
         * @throws PtlException
         * @throws Exception
         */
        public static PtlRect setRectCoordinate(PtlRect outputRect)
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            try
            {
                while (!isValueOkay)
                {
                    Console.WriteLine("指定する矩形の各数値を入力してください。");
                    Console.Write("top (mm) : ");
                    top = float.Parse(Console.ReadLine());
                    Console.Write("bottom (mm) : ");
                    bottom = float.Parse(Console.ReadLine());
                    if (top < bottom)
                    {
                        //不正矩形は再入力させる
                        Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                        continue;
                    }
                    Console.Write("left (mm) : ");
                    left = float.Parse(Console.ReadLine());
                    Console.Write("right (mm) : ");
                    right = float.Parse(Console.ReadLine());
                    if (right < left)
                    {
                        //不正矩形は再入力させる
                        Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                        continue;
                    }

                    //矩形を正しく指定できた場合の処理
                    isValueOkay = true;
                    outputRect.setLeft(left);
                    outputRect.setBottom(bottom);
                    outputRect.setRight(right);
                    outputRect.setTop(top);
                }
            }
            catch (IOException ex)
            {
                throw ex;
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
            return outputRect;
        }

        /**
         * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
         * 
         * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
         * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
         * @return argsの数値を読み取った結果を戻す
         * @throws ArgumentException argsが0か1でなかった場合に発生。
         */
        public static bool readBoolArgs(String args, String errorMessage)
        {
            bool trueOrFalse = false;
            try
            {
                // argsの読み取り
                switch (args)
                {
                    case "0":
                        trueOrFalse = false;
                        break;
                    case "1":
                        trueOrFalse = true;
                        break;
                    default:
                        throw new ArgumentException(errorMessage);
                }
            }
            catch (ArgumentException ex)
            {
                throw ex;
            }
            return trueOrFalse;
        }

        // 文字列を PtlContent.ALIGN の数値型として返す
        static PtlContent.ALIGN valueAlign(String textAlign)
        {
            if (textAlign.Equals(PtlContent.ALIGN.ALIGN_BOTTOM.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_BOTTOM;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_BOTTOM_LEFT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_BOTTOM_LEFT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_BOTTOM_RIGHT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_BOTTOM_RIGHT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_CENTER.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_CENTER;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_LEFT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_LEFT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_RIGHT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_RIGHT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_TOP.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_TOP;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_TOP_LEFT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_TOP_LEFT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_TOP_RIGHT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_TOP_RIGHT;
            }
            else
            {
                throw new ArgumentException();
            }
        }
    }
}
