﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：テキストボックスを用いて指定したページの指定した位置に
        テキストを挿入します。
  　　　配置矩形のみ、プログラム内で指定します。
  　　　行間の大きさ、文字間隔の大きさを数値で指定します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;
using System.IO;

namespace TBoxDrawWithSpacing
{
    internal class TBoxDrawWithSpacing
    {
        // そのクラスのusageを表示する関数
        private static void printUsage()
        {
            Console.WriteLine("usage: TBoxDrawWithSpacing in-pdf-file out-pdf-file page-num text-to-add text-align"
                    + "line-spacing-num" + "char-spacing-num");
            Console.WriteLine("line-spacing-num : 行間の設定。フォントサイズに対する倍数で、"
                             + "-1～3の値で設定（小数可）");
            Console.WriteLine("char-spacing-num : 文字間隔の設定。フォントサイズに対する倍数で、"
                             + "-1～2の値で設定（小数可）");
        }

        static void Main(string[] args)
        {
            if (args.Length < 7)
            {
                printUsage();
                return;
            }

            //line-spacing-num、char-spacing-numの引数の判定
            float lineSpacingNum = float.Parse(args[5]);
            if (lineSpacingNum < -1 || 3 < lineSpacingNum)
            {
                Console.WriteLine("ERROR:line-spacing-numは-1～3の数値で設定して下さい。");
                printUsage(); // usageメッセージの表示
                return;
            }

            float charSpacingNum = float.Parse(args[6]);
            if (charSpacingNum < -1 || 2 < charSpacingNum)
            {
                Console.WriteLine("ERROR:char-spacing-numは-1～2の数値で設定して下さい。");
                printUsage(); // usageメッセージの表示
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);
                    int pageToAdd = int.Parse(args[2]);
                    int numPages = doc.getPageCount();
                    Console.WriteLine("ページ数：" + numPages);
                    if ((numPages < 0) || (numPages < pageToAdd))
                    {
                        TextWriter errorWriter = Console.Error;
                        errorWriter.WriteLine("テキスト挿入ページは全ページ数よりも小さい正の値を指定してください。");
                        return;
                    }

                    //引数の読み取り
                    String textToAdd = args[3];
                    String textAlign = args[4];

                    //ページコンテナの取得
                    using (PtlPages pages = doc.getPages())
                    {
                        // ページコンテナが空かどうか
                        if (pages.isEmpty())
                        {
                            TextWriter errorWriter = Console.Error;
                            errorWriter.WriteLine("ページコンテナが空");
                            return;
                        }

                        try
                        {
                            using (PtlPage page = pages.get(pageToAdd - 1)) // ページの取得(パラメータindexは0が先頭のため1を引く)
                            using (PtlContent content = page.getContent())  // ページコンテントの取得
                            using (PtlRect outputRect = new PtlRect())      // 出力矩形の初期化
                            using (PtlTextBox textBox = content.drawTextBox(setRectCoordinate(outputRect), valueAlign(textAlign), 210, 297))
                            // 文字描画のパラメータクラス。
                            using (PtlParamWriteStringTextBox paramSpacing = new PtlParamWriteStringTextBox())
                            {
                                paramSpacing.setLineSpacing(lineSpacingNum);    //行間の設定
                                paramSpacing.setCharSpacing(charSpacingNum);    //文字間隔の設定

                                // 文字列出力
                                textBox.writeString(textToAdd, paramSpacing);
                                textBox.terminate();
                            }
                        }
                        catch (ArgumentException ex)
                        {
                            //PtlContent.ALIGN指定のエラー処理
                            Console.WriteLine(ex.Message);
                            Console.WriteLine("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                            Console.WriteLine(ex.StackTrace);
                        }
                    }

                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        /**
         * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
         * 原点はPDFの左下端。
         * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
         * 特に外部からの呼び出しを想定しないためprivateとする。
         * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
         * @return 指定したleft, bottom, right, topの数値を持つPtlRect
         * @throws IOException
         * @throws PtlException
         * @throws Exception
         */
        public static PtlRect setRectCoordinate(PtlRect outputRect)
        {
            float top, bottom, left, right;
            bool isValueOkay = false;
            try
            {
                while (!isValueOkay)
                {
                    Console.WriteLine("指定する矩形の各数値を入力してください。");
                    Console.Write("top (mm) : ");
                    top = float.Parse(Console.ReadLine());
                    Console.Write("bottom (mm) : ");
                    bottom = float.Parse(Console.ReadLine());
                    if (top < bottom)
                    {
                        //不正矩形は再入力させる
                        Console.WriteLine("topの値はbottomよりも大きい値を指定して再度入力してください。");
                        continue;
                    }
                    Console.Write("left (mm) : ");
                    left = float.Parse(Console.ReadLine());
                    Console.Write("right (mm) : ");
                    right = float.Parse(Console.ReadLine());
                    if (right < left)
                    {
                        //不正矩形は再入力させる
                        Console.WriteLine("rightの値はleftよりも大きい値を指定して再度入力してください。");
                        continue;
                    }

                    //矩形を正しく指定できた場合の処理
                    isValueOkay = true;
                    outputRect.setLeft(left);
                    outputRect.setBottom(bottom);
                    outputRect.setRight(right);
                    outputRect.setTop(top);
                }
            }
            catch (IOException ex)
            {
                throw ex;
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
            return outputRect;
        }

        // 文字列を PtlContent.ALIGN として返す
        static PtlContent.ALIGN valueAlign(String textAlign)
        {
            if (textAlign.Equals(PtlContent.ALIGN.ALIGN_BOTTOM.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_BOTTOM;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_BOTTOM_LEFT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_BOTTOM_LEFT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_BOTTOM_RIGHT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_BOTTOM_RIGHT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_CENTER.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_CENTER;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_LEFT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_LEFT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_RIGHT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_RIGHT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_TOP.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_TOP;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_TOP_LEFT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_TOP_LEFT;
            }
            else if (textAlign.Equals(PtlContent.ALIGN.ALIGN_TOP_RIGHT.ToString()))
            {
                return PtlContent.ALIGN.ALIGN_TOP_RIGHT;
            }
            else
            {
                throw new ArgumentException();
            }
        }
    }
}
