﻿/*
  Antenna House PDF Tool API V7.0
  .Net Interface sample program

  概要：透かしに設定したテキストのページ内での配置位置を設定します
        透かしを設定します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

using PdfTkNet;
using System;

namespace TextWaterMarkSetTextAlign
{
    internal class TextWaterMarkSetTextAlign
    {
        // そのクラスのusageを表示する関数
        public static void printUsage()
        {
            Console.Write("usage: TextWaterMarkSetTextAlign in-pdf-file out-pdf-file");
            Console.WriteLine("set-text-align");
            Console.WriteLine("set-text-align : ");
            Console.WriteLine("0 : 左寄せ  1 : 中央寄せ  2 : 右寄せ");
        }

        static void Main(string[] args)
        {
            if (args.Length < 3)
            {
                printUsage(); // usageメッセージの表示
                return;
            }

            // コマンドライン引数の読み取り]
            String textAlign = "";
            try
            {
                textAlign = args[2];
            }
            catch (ArgumentException ex)
            {
                Console.WriteLine(ex.Message);
                printUsage(); // usageメッセージの表示
                return;
            }

            try
            {
                using (PtlParamInput inputFile = new PtlParamInput(args[0]))
                using (PtlParamOutput outputFile = new PtlParamOutput(args[1]))
                using (PtlPDFDocument doc = new PtlPDFDocument())
                {
                    // PDFファイルをロードします。
                    doc.load(inputFile);
                    // 透かしの追加
                    appendTextWaterMarkSetTextAlign(doc, textAlign);
                    // ファイルに保存します。
                    doc.save(outputFile);
                }
            }
            catch (PtlException pex)
            {
                Console.WriteLine("PtlException : ErrorCode = " + pex.getErrorCode() +
                                   "\n  " + pex.getErrorMessage());
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                Console.WriteLine(ex.StackTrace);
            }
            finally
            {
                Console.WriteLine("-- 完了 --");
            }
        }

        public static void appendTextWaterMarkSetTextAlign(PtlPDFDocument doc, String textAlign)
        {
            try
            {
                using (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText())
                {
                    switch (textAlign)
                    {
                        case "0"://左寄せ
                            waterMarkText.setTextAlign(PtlParamWaterMarkText.TEXT_ALIGN.TEXT_ALIGN_LEFT);
                            break;
                        case "1"://中央寄せ
                            waterMarkText.setTextAlign(PtlParamWaterMarkText.TEXT_ALIGN.TEXT_ALIGN_CENTER);
                            break;
                        case "2"://右寄せ
                            waterMarkText.setTextAlign(PtlParamWaterMarkText.TEXT_ALIGN.TEXT_ALIGN_RIGHT);
                            break;
                        default:
                            throw new ArgumentException("set-text-alignは0から2の整数で入力して下さい");
                    }
                    // setTextAlign()の設定が正常に動くためにはAcrobatCompatibleをtrueにする必要がある。
                    waterMarkText.setAcrobatCompatible(true);

                    // 複数行で透かしを表示するためには傾きが0度である必要がある
                    waterMarkText.setTextAngle(0.0f);

                    //setAcrobatCompatibleをtrueにしたときにPDF出力ができるため、以下の設定をする
                    // PtlParamWaterMark.setCustomPageRange()では不連続なページを指定しない
                    // PtlParamWaterMark.setMargin()を設定しない
                    // PtlParamWaterMark.setRect()を設定しない
                    // PtlParamWaterMark.setTiling()をfalseにする
                    waterMarkText.setTiling(false);
                    // PtlParamWaterMarkText.setOutlineColor()を設定しない
                    // PtlParamWaterMarkText.setWriteDiagonal()をfalseにする
                    waterMarkText.setWriteDiagonal(false);

                    // 以下、その他の項目について透かしであることが分かりやすいようデフォルトの設定をする
                    // 透かしの名前の設定
                    waterMarkText.setName("waterMarkText");
                    // 透かしの配置の設定
                    waterMarkText.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
                    // 透かしのZオーダーの設定 ZORDER_FRONT = 1 
                    waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
                    // 透かしの不透明度の設定
                    waterMarkText.setOpacity(1.0f);
                    // 透かしに指定する文字列の設定
                    waterMarkText.setString("WaterMarkText line1\n Another Line watermarkText line2\n line3");
                    // 透かしに指定するフォントの設定
                    using (PtlParamFont font
                        = new PtlParamFont("Times New Roman", 40.0f, PtlParamFont.WEIGHT.WEIGHT_MEDIUM, false, true))
                    {
                        waterMarkText.setFont(font);
                    }
                    // 透かしの文字に指定する色の設定
                    using (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f))
                    {
                        waterMarkText.setTextColor(colorText);
                    }
                    // デフォルトの設定終わり

                    // 透かしを文書に設定する
                    doc.appendWaterMark(waterMarkText);
                }
            }
            catch (PtlException ex)
            {
                throw ex;
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
    }
}
