/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：指定したページの指定した位置にPDFの1ページを挿入します。
  　　　挿入する際に指定した角度に傾けます。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;


public class AddPdfOnPageSetAngle {
    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.print("usage: java AddForm in-pdf-file out-pdf-file");
        System.out.println("page-num insert-pdf-file insert-page-num align angle");
    }
    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 7) {
            printUsage(); // usageメッセージの表示
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            //コマンドライン引数の取得
            int pageToAdd = Integer.parseInt(args[2]);
            String insertPdfURI = args[3];
            int insertPageNum = Integer.parseInt(args[4]);
            String align = args[5];
            float angle = Float.parseFloat(args[6]);
            // pageToAddの判定
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);
            if((numPages < 0)||(numPages < pageToAdd)) {
                System.err.println("page-numは入力PDFの全ページ数よりも小さい正の値を" +
                                   "指定してください。");
                printUsage();
                return;
            }


            try (PtlPages pages = doc.getPages()) { // ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ページコンテナが空");
                    return;
                }
                // 書き込み先ページを取得(パラメータindexは0が先頭のため1を引く)
                try (PtlPage page = pages.get(pageToAdd - 1)) {
                    //フォーム（PDFのページ）の描画
                    drawFormSetAngle(page, insertPdfURI, insertPageNum, align, angle,  br);
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void drawFormSetAngle(PtlPage page, String insertPdfURI,
                                        int insertPageNum, String align, float angle,
                                        BufferedReader br)
                       throws PtlException, Exception, Error {
        try(PtlContent content = page.getContent();// 挿入先ページコンテントの取得
            PtlParamInput insertPdf = new PtlParamInput(insertPdfURI); // 挿入PDF指定に使うパラメータクラス
            PtlPDFDocument docToInsert = new PtlPDFDocument(); // 挿入PDFの実体
            PtlRect outputRect = new PtlRect()) {// 出力矩形の初期化
            docToInsert.load(insertPdf);
            int numPages = docToInsert.getPageCount();
            if((numPages < 0)||(numPages < insertPageNum)) {
                System.err.println("insert-page-numは挿入するPDFの全ページ数よりも小さい正の値を指定してください。");
                System.out.println("挿入PDFページ数：" + numPages);
                return;
            }
            
            try(PtlPages pagesToInsert = docToInsert.getPages();
                PtlPage pageToInsert = pagesToInsert.get(insertPageNum - 1);
                PtlParamDrawForm formInsert = new PtlParamDrawForm()) {
                try {
                    // ページの取得(パラメータindexは0が先頭のため1を引く)
                    formInsert.setPage(pageToInsert);
                    formInsert.setAngle(angle);
                    // 指定ページの挿入
                    content.drawForm(setRectCoordinate(br, outputRect), PtlContent.ALIGN.valueOf(align), formInsert);
                }
                catch (IllegalArgumentException ex) {//PtlContent.ALIGN指定のエラー処理
                    System.out.println(ex.getMessage());
                    System.out.println("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                    ex.printStackTrace();
                }
            }
        }
    }
    
    /**
     * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
     * 原点はPDFの左下端。
     * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * @param br BufferedReader。数値の読み取りに使う。
     * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
     * @return 指定したleft, bottom, right, topの数値を持つPtlRect
     * @throws IOException
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    public static PtlRect setRectCoordinate(BufferedReader br, PtlRect outputRect)
           throws IOException, PtlException, Exception, Error {
        float top, bottom, left, right;
        boolean isValueOkay = false;
        
        while(!isValueOkay) {
            System.out.println("指定する矩形の各数値を入力してください。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) {//不正矩形は再入力させる
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            if(right < left) {//不正矩形は再入力させる
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setLeft(left);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setTop(top);
        }
        return outputRect;
    }
}