/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：公開キーによるセキュリティの解除
　PKCS#12公開キーでPDFを開き、セキュリティを解除して新規PDFとして保存します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class DecryptPubkey {
    
    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java DecryptPubkey in-pdf-file out-pdf-file"
                + "PKCS#12-path pkcs#12-password");
    }    

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 2) {
            printUsage();
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamInput pkcs12Pubkey = new PtlParamInput(args[2]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            
            doc.setPassword(args[3]);
            doc.setPKCS12(pkcs12Pubkey);
            // PDFファイルをロードします。
            doc.load(inputFile);


            // 暗号化の取得
            if (doc.isEncrypted()) {
                if(hasPubkeyFullPermission(doc)) {//trueの場合のみにセキュリティを解除できる。
                    doc.removeEncrypt();  //暗号化情報の削除
                    doc.save(outputFile); //ファイルの保存
                }
                else {
                    System.out.println("このPKCS#12ファイルにはPDFの暗号化を解除する権限がありません。"
                            + "証明書セキュリティの削除を中断し、プログラムを終了します。");
                }
            }
            else {
                System.out.println("暗号化されていないファイルです");
            }
            

        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
    
    
    /**
     * 引数で受け取ったPDFファイルの証明書セキュリティがすべての権限を許可されているかを
     * 判別します。
     * 
     * @param doc 判別したいPDF文書をあらわすクラス
     * @return PDFを開いた証明書がすべての権限を許可されているかどうか
     * @throws PtlException
     * @throws Exception 証明書セキュリティがかけられていない場合にthrowする。
     * @throws Error 
     */
    public static boolean hasPubkeyFullPermission(PtlPDFDocument doc)
               throws PtlException, Exception, Error {
        boolean hasFullPerm = false;
        try (PtlEncrypt encrypt = doc.getEncrypt()) {
            if(encrypt.getFilterType() == PtlEncrypt.FILTER_TYPE.FILTER_PUBKEY) {
                try(PtlEncryptPubKey encryptPubkey = (PtlEncryptPubKey)encrypt;
                    PtlEncryptPermissionPubKey permsPubkey = (PtlEncryptPermissionPubKey)encryptPubkey.getPermission()) {
                    hasFullPerm = permsPubkey.hasFullPermission();
                }
            }
            else {
                throw new Exception("このPDFにかけられたセキュリティは証明書セキュリティではありません。"
                                    + "処理を中断します。");
            }
        }
        
        return hasFullPerm;
    }
}