/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：公開鍵を用いてPDFを暗号化する際に全ての権限で許可を出す
  (X.509ファイルで暗号化)

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;

public class EncryptSetPubKeySetFullPermission {
    
    // そのクラスのusageを表示する関数
    private static void printUsage() {
            System.out.print("usage: java EncryptSetPubKeyWithX509 in-pdf-file out-pdf-file ");
            System.out.println("暗号化種類 X509Pubkey-path set-full-permission");
            System.out.println("暗号化種類");
            System.out.println("0 : 128 bit RC4  1 : 128 bit AES  2 : 256 bit AES");
            System.out.println("set-full-permission");
            System.out.println("0 : デフォルトのアクセス権限を設定する。  1 : 全ての権限を許可する");
    }
    
    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            printUsage();
            return;
        }

        // コマンドライン引数の取得
        int kind;
		boolean setFullPerm;
        try {
            kind =  Integer.parseInt(args[2]);
            switch (kind) {
            case 0:
            case 1:
            case 2:
                break;
            default:
                System.out.println("暗号化種類には、0から2の数値を指定してください。");
                printUsage(); // usageメッセージの表示
                return;
            }
        }
        catch(NumberFormatException numfe) {
            System.out.println("ERROR!!暗号化種類の文字形式が正しくありません。");
            System.out.println("暗号化種類には、0から2の数値を指定してください。");
            numfe.printStackTrace();
            printUsage(); // usageメッセージの表示
            return;
        }

        try {
            setFullPerm = readBoolArgs(args[4], "set-full-permissionは" +
                                           "0か1で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }
		
        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlParamInput x509Pubkey = new PtlParamInput(args[3]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            
            // PDFファイルをロードします
            doc.load(inputFile);
            
            // PDFにX.509形式の証明書を設定します
            switch (kind) {
            case 0:
                // 128 bit RC4
                System.out.println("128bit RC4で暗号化します。");
                encryptX509PubkeySetFullPerm128RC4(doc, x509Pubkey, setFullPerm);
                break;
            case 1:
                // 128 bit AES
                System.out.println("128bit AESで暗号化します。");
                encryptX509PubkeySetFullPerm128AES(doc, x509Pubkey, setFullPerm);
                break;
            case 2:
                // 256 bit AES
                System.out.println("256bit AESで暗号化します。");
                encryptX509PubkeySetFullPerm256AES(doc, x509Pubkey, setFullPerm);
                break;
            }

            // ファイルに保存します
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
    
    
    private static void encryptX509PubkeySetFullPerm128RC4(PtlPDFDocument doc, PtlParamInput x509Pubkey, boolean setFullPerm)
            throws PtlException, IOException, Exception, Error {
        try(PtlEncryptPubKey128RC4 encPubkey128RC4 = new PtlEncryptPubKey128RC4()) {//256ビットAES暗号化された公開キーセキュリティハンドラを表すクラス
            encryptX509PubkeySetFullPerm(doc, x509Pubkey, encPubkey128RC4, setFullPerm);
        }
    }    
    
    private static void encryptX509PubkeySetFullPerm128AES(PtlPDFDocument doc, PtlParamInput x509Pubkey, boolean setFullPerm)
            throws PtlException, IOException, Exception, Error {
        try(PtlEncryptPubKey128AES encPubkey128AES = new PtlEncryptPubKey128AES()) {//256ビットAES暗号化された公開キーセキュリティハンドラを表すクラス
            encryptX509PubkeySetFullPerm(doc, x509Pubkey, encPubkey128AES, setFullPerm);
        }
    }    
    
    private static void encryptX509PubkeySetFullPerm256AES(PtlPDFDocument doc, PtlParamInput x509Pubkey, boolean setFullPerm)
            throws PtlException, IOException, Exception, Error {
        try(PtlEncryptPubKey256AES encPubkey256AES = new PtlEncryptPubKey256AES()) {//256ビットAES暗号化された公開キーセキュリティハンドラを表すクラス
            encryptX509PubkeySetFullPerm(doc, x509Pubkey, encPubkey256AES, setFullPerm);
        }
    }
    
    
    private static void encryptX509PubkeySetFullPerm(PtlPDFDocument doc, PtlParamInput x509Pubkey,
                                          PtlEncryptPubKey encPubkey, boolean setFullPerm)
            throws PtlException, IOException, Exception, Error {
        try(PtlRecipients recipients = encPubkey.getRecipients();   //証明書受信者のコンテナを表したクラス
            PtlRecipient recipient = new PtlRecipient();            //証明書の受信者を表したクラス
            PtlEncryptPermissionPubKey permission = new PtlEncryptPermissionPubKey()) {
            //すべての編集権限を与えるかどうかを設定
            permission.setFullPermission(setFullPerm);

			//受信者クラスにX.509形式の証明書を設定
            recipient.setX509(x509Pubkey);
            //受信者クラスにアクセス許可フラグを設定
            recipient.setPermission(permission);
            //受信者コンテナのクラスに受信者クラスを追加
            recipients.append(recipient);
            
            //以下は標準セキュリティハンドラと共通の設定
            //すべてのコンテンツを暗号化する
            encPubkey.setEncryptComponent(PtlEncrypt.ENCRYPT_COMPONENT.ENCRYPT_ALL);

            // 設定したセキュリティハンドラを用いてPDFを暗号化
            doc.setEncrypt(encPubkey);
        }
    }
        
    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
        throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }
}