/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：座標ベースでのテキスト抽出の際に同じテキストが
        どの程度重なり合っていたら除外するかを指定します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.BufferedWriter;
import java.io.PrintWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.charset.StandardCharsets;

import jp.co.antenna.ptl.*;

public class ExtTextSetSameTextOmitRatio {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java ExtTextSetSameTextOmitRatio"
                         + " in-pdf-file out-text-file"
                         + " page-to-extract sameText-omit-ratio");
        System.out.println("sameText-omit-ratio:");
        System.out.println("同じ文字が重なり合っていた場合にどの程度重なり合っていたら除外するかを小数で指定");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の読み取り・判定
        // 出力PDFの名前はあとで渡すためにString型で保存する。
        String outputTextURI = args[1];
        int pageToExtract = Integer.parseInt(args[2]);
        float sameTextOmitRatio = Float.parseFloat(args[3]);

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロード
            doc.load(inputFile);
            //ページ数を取得
            int wholePageNum = doc.getPageCount();
            //pageToExtractのエラー処理
            if(wholePageNum < pageToExtract) {
                System.out.println("ERROR: page-to-extractはPDFの総ページ数より"+
                                   "小さい値を指定してください。");
                System.out.println("総ページ数：" + wholePageNum);
                printUsage();
                return;
            }
            
            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ERROR : ページコンテナが空");
                    return;
                }

                // ページの取得(パラメータindexは0が先頭のため1を引く)
                try (PtlPage page = pages.get(pageToExtract - 1);
                     // ページコンテントの取得
                     PtlContent content = page.getContent();
                     // 文字抽出のパラメータクラス。
                     PtlParamExtractText paramExtractText = new PtlParamExtractText()) {
                    // 同じテキストがどのぐらい重なっていたら取り除くかを指定する。
                    paramExtractText.setSameTextOmitRatio(sameTextOmitRatio);
                    
                    // 抽出するテキストを座標順に並べるよう指定
                    //（setSameTextOmitRatioのためには指定が必要）
                    paramExtractText.setTextType(PtlParamExtractText.TEXT_TYPE.TEXT_SORT);

                    // 文字列抽出
                    String TextFromPdf = content.extractText(paramExtractText);
                    // 文字列の出力
                    outputTextFile(outputTextURI, TextFromPdf);
                    System.out.println(TextFromPdf);
                }
            }
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }


    /**
     * テキストファイルを出力するための関数。 
     * 出力エンコードはUTF-8を指定する。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * 
     * @param outputTextURI 出力ファイルのURI。
     * @param TextFromPdf 出力したいString型変数
     */
    private static void outputTextFile(String outputTextURI, String TextFromPdf) {
        System.out.println("Output text URI :" + outputTextURI);
        try(BufferedWriter bw = Files.newBufferedWriter(Paths.get(outputTextURI),
                                                        StandardCharsets.UTF_8);
            PrintWriter pw = new PrintWriter(bw, false)) {	
            pw.print(TextFromPdf);
        }
        catch(IOException e) {
            System.out.println("IOException occured!!");
            e.printStackTrace();
        }
    }
}