/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：テキスト抽出のソート順を指定します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.BufferedWriter;
import java.io.PrintWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.charset.StandardCharsets;

import jp.co.antenna.ptl.*;

public class ExtTextWithRplcReplacementCharacterToSpace {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java ExtTextWithRplcReplacementCharacterToSpace"
                         + " in-pdf-file out-text-file"
                         + " page-to-extract rplc-replacement-character");
        System.out.println("--以下の文字に対して、[0:スペースに置き換えない  1:置き換える]を選択--");
        System.out.println("rplc-replacement-character : 置換文字");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の読み取り・判定
        // 出力PDFの名前はあとで渡すためにString型で保存する。
        String outputTextURI = args[1];
        int pageToExtract = Integer.parseInt(args[2]);
        boolean rplcReplacementCharacter;
        int unicodeCharFlag = 0;
        
        //各種削除フラグの読み取り判定
        //rplc-replacement-character
        try {
            rplcReplacementCharacter = readBoolArgs(args[3],
                                                   "rplc-replacement-characterは" +
                                                   "0か1で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }
        
        //各フラグの論理和をunicodeCharFlagに設定
        if(rplcReplacementCharacter) {
            unicodeCharFlag = unicodeCharFlag | PtlParamExtractText.AHEXTRACTTEXT_UNI_REPLACEMENT_CHARACTER;
        }


        try(PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロード
            doc.load(inputFile);
            try(PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if(pages.isEmpty()) {
                    System.out.println("ERROR : ページコンテナが空");
                    throw new Error("ERROR : ページコンテナが空");
                }
                //ページ数を取得
                int wholePageNum = doc.getPageCount();
                
                //pageToExtractが0ならすべてのページを処理する
                if(pageToExtract == 0) {
                    StringBuilder wholeTextFromPdf = new StringBuilder();
                    // ページの取得(パラメータindexは0が先頭のため1を引く)
                    for(int i = 0; i < wholePageNum; i++) {
                        try(PtlPage page = pages.get(i);
                            PtlContent content = page.getContent()) { // ページコンテントの取得
                            System.out.println((i+1)  + "ページ目のテキストを抽出します。");
                            wholeTextFromPdf.append(extractTextSetReplaceUnicodeFlag(unicodeCharFlag, content));
                        }
                    }
                    outputTextFile(outputTextURI, wholeTextFromPdf.toString());
                }else {
                    if(wholePageNum < pageToExtract) { //pageToExtractのエラー処理
                        System.out.println("ERROR: page-to-extractはPDFの総ページ数より"+
                                           "小さい値を指定してください。");
                        System.out.println("総ページ数：" + wholePageNum);
                        printUsage();
                        throw new Error("ERROR: page-to-extractはPDFの総ページ数より"+
                                        "小さい値を指定してください。");
                    }
                    try(PtlPage page = pages.get(pageToExtract - 1); //指定したページを取得する
                        PtlContent content = page.getContent()) {
                         System.out.println(pageToExtract  + "ページ目のテキストを抽出します。");
                        outputTextFile(outputTextURI, extractTextSetReplaceUnicodeFlag(unicodeCharFlag, content));
                    }
                }
            }
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    
    private static String extractTextSetReplaceUnicodeFlag(int unicodeCharFlag, PtlContent content)
                        throws IOException, PtlException, Exception, Error {
        try(PtlParamExtractText paramExtractText = new PtlParamExtractText()) { // 文字抽出のパラメータクラス。
            //setUnicodeToRemoveをパラメータに設定
            paramExtractText.setUnicodeToSpace(unicodeCharFlag);
            // 文字列抽出
            String textFromPdf = content.extractText(paramExtractText);
            System.out.println(textFromPdf);

            // 抽出した文字列を返す
            return textFromPdf;
        }
    }

    /**
     * テキストファイルを出力するための関数。 
     * 出力エンコードはUTF-8を指定する。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * 
     * @param outputTextURI 出力ファイルのURI。
     * @param TextFromPdf 出力したいString型変数
     */
    private static void outputTextFile(String outputTextURI, String TextFromPdf){
        System.out.println("Output text URI :" + outputTextURI);
        try(BufferedWriter bw = Files.newBufferedWriter(Paths.get(outputTextURI),
                                                        StandardCharsets.UTF_8);
            PrintWriter pw = new PrintWriter(bw, false)) {	
            pw.print(TextFromPdf);
        }
        catch(IOException e) {
            System.out.println("IOException occured!!");
            e.printStackTrace();
        }
    }


    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
                          throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }
}