/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要： JPEG画像の書き出し時にパススルー設定をするかどうかを選択します。
        指定されたインデックス番号の画像エレメントを取得し、
  　　　画像エレメントからファイルを書き出します。
  　　　

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;


public class ExtractImageSetPassthrough {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java ExtractImageSetPassthrough in-pdf-file out-image-body" +
                           " page-to-extract image-index-num set-passthrough");
        System.out.println("set-passthrough : ");
        System.out.println("0 : JPEG圧縮する  1 : JPEGは加工せずにそのまま出力する");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 5) {
            printUsage();
            return;
        }

        // コマンドライン引数の取得
        int pageToExtract = Integer.parseInt(args[2]);
        int imageIndexNum = Integer.parseInt(args[3]);
        boolean setPassthrough = false;
        try {
            setPassthrough = readBoolArgs(args[4], "set-passthroughは 0か1で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }
        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlPages pages = doc.getPages()) { // ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ページコンテナが空");
                    return;
                }

                // それ以外であれば指定ページ番号を検索。
                int wholePageNum = doc.getPageCount();
                //pageToSearchのエラー処理
                if(wholePageNum < pageToExtract) {
                    System.out.println("ERROR: page-to-extractはPDFの総ページ数より"+
                                       "小さい値を指定してください。");
                    System.out.println("総ページ数：" + wholePageNum);
                    printUsage();
                    return;
                }
                // ページの取得(index番号は0が先頭のため1を引く)
                try (PtlPage page = pages.get(pageToExtract - 1)) {
                    // 画像抽出
                    extractImageSetPassthrough(page, args[1], imageIndexNum, setPassthrough);
                }
            }
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void extractImageSetPassthrough(PtlPage page, String imagefileName,
                                          int imageIndexNum, boolean setPassthrough)
        throws PtlException, Exception, Error {
        // ページコンテント・画像エレメントの取得
        try (PtlContent content = page.getContent();
             PtlEditElements elems = content.getEditElements(PtlContent.GET_IMAGE)) {
            int numElems = elems.getCount();
            if (numElems == 0) {
                System.out.println("指定されたページに画像エレメントがありませんでした。");
                return; // 画像エレメントが無い場合はreturnする
            }
            //imageIndexNumのエラー処理
            if(numElems < imageIndexNum) {
                System.out.println("ERROR: image-index-numは対象ページの総画像数より"+
                                   "小さい値を指定してください。");
                System.out.println("総画像数：" + numElems);
                printUsage();
                return;
            }


            // 画像エレメントの取得
            try (PtlEditElement elem = elems.get(imageIndexNum - 1)) {
                PtlEditElement.ELEMENT_TYPE type = elem.getType();
                switch (type) {
                case TYPE_IMAGE:
                    PtlEditImage elemImage = (PtlEditImage)elem;
                    elemImage.setPathThrough(setPassthrough);
                    try (PtlParamOutput outputImage = new PtlParamOutput(imagefileName)) {
                        // 画像出力(パススルー設定を確かめるため、FORMAT_JPEGで出力する。)
                        elemImage.writeFile(outputImage, PtlEditImage.OUTPUT_FORMAT.FORMAT_JPEG);
                    }
                    break;
                default:
                    System.out.println("指定されたエレメントは画像ではありません。"
                                     + "処理を中断します。");
                    break;
                }
            }
        }
    }
    
    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
        throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }    
}