/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：指定した矩形からテキスト抽出

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.BufferedWriter;
import java.io.PrintWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.charset.StandardCharsets;

import jp.co.antenna.ptl.*;

public class ExtractTextSetRect {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java ExtractTextSetRect in-pdf-file out-text-file" + 
                           " page-to-extract overlap-ratio ignore-actual-text");
        System.out.println("ignore-actual-text:");
        System.out.println("0:ActualTextを無視しない 1:ActualTextを無視する");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 5) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の読み取り・判定
        // 出力PDFの名前はあとで渡すためにString型で保存する。
        String outputTextURI = args[1];
        int pageToExtract = Integer.parseInt(args[2]);
        float overlapRatio = Float.parseFloat(args[3]);
        boolean ignoreActualText = true;
        try {
            ignoreActualText = readBoolArgs(args[4],
                                       "ignore-actual-textは1か0で指定してください。");
                                       
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロード
            doc.load(inputFile);
            //ページ数を取得
            int wholePageNum = doc.getPageCount();
            //pageToExtractのエラー処理
            if(wholePageNum < pageToExtract) {
                System.out.println("ERROR: page-to-extractはPDFの総ページ数より"+
                                   "小さい値を指定してください。");
                System.out.println("総ページ数：" + wholePageNum);
                printUsage();
                return;
            }
            
            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ERROR : ページコンテナが空");
                    return;
                }

                // ページの取得(パラメータindexは0が先頭のため1を引く)
                try (PtlPage page = pages.get(pageToExtract - 1);
                     // ページコンテントの取得
                     PtlContent content = page.getContent();
                     // 文字抽出のパラメータクラス。
                     PtlParamExtractText paramExtractText = new PtlParamExtractText();
                     PtlRect outputRect = new PtlRect()) {
                    // テキスト抽出する矩形を指定
                    paramExtractText.appendRect(setRectCoordinate(br, outputRect));
                    // テキストが矩形とどのくらい重なっていれば抽出するか設定
                    paramExtractText.setTextOverlapRatio(overlapRatio);
                    // ActialTextを無視するかどうか
                    paramExtractText.setIgnoreActualText(ignoreActualText);

                    // 文字列抽出
                    String TextFromPdf = content.extractText(paramExtractText);

                    // 文字列の出力
                    outputTextFile(outputTextURI, TextFromPdf);
                    System.out.println(TextFromPdf);
                }
            }

        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }


    /**
     * テキストファイルを出力するための関数。 
     * 出力エンコードはUTF-8を指定する。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * 
     * @param outputTextURI 出力ファイルのURI。
     * @param TextFromPdf 出力したいString型変数
     */
    private static void outputTextFile(String outputTextURI, String TextFromPdf) {
        System.out.println("\nOutput text URI :" + outputTextURI +"\n");
        try(BufferedWriter bw = Files.newBufferedWriter(Paths.get(outputTextURI),
                                                        StandardCharsets.UTF_8);
            PrintWriter pw = new PrintWriter(bw, false)) {	
            pw.print(TextFromPdf);
        }
        catch(IOException e) {
            System.out.println("IOException occured!!");
            e.printStackTrace();
        }
    }

    /**
     * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
     * 原点はPDFの左下端。
     * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * @param br BufferedReader。数値の読み取りに使う。
     * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
     * @return 指定したleft, bottom, right, topの数値を持つPtlRect
     * @throws IOException
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    public static PtlRect setRectCoordinate(BufferedReader br, PtlRect outputRect)
           throws IOException, PtlException, Exception, Error {
        float top, bottom, left, right;
        boolean isValueOkay = false;
        
        while(!isValueOkay) {
            System.out.println("指定する矩形の各数値を入力してください。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) {//不正矩形は再入力させる
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            if(right < left) {//不正矩形は再入力させる
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setLeft(left);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setTop(top);
        }
        return outputRect;
    }


    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
        throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }
}